#!/bin/bash
#
# Setup script for local Darwin PMDA development
# Creates a minimal local PCP environment for dbpmda testing
# No system installation required!
#
# Usage: ./setup-local-pcp.sh
#
# After running this script, use dev-test.sh for rapid development cycles.

set -e

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
REPO_ROOT="$(cd "$SCRIPT_DIR/../.." && pwd)"
LOCAL_VAR="$SCRIPT_DIR/local-pcp-var"

# Find the PCP build directory (Makepkgs creates pcp-X.Y.Z)
BUILD_DIR=""
for dir in "$REPO_ROOT"/pcp-*; do
    if [ -d "$dir" ]; then
        BUILD_DIR="$dir"
        break
    fi
done

if [ -z "$BUILD_DIR" ]; then
    echo "Error: Cannot find Makepkgs build directory (pcp-X.Y.Z)"
    echo "Please run: cd $REPO_ROOT && ./Makepkgs --verbose"
    exit 1
fi

echo "=== Setting up Local PCP Development Environment ==="
echo "Build directory: $BUILD_DIR"
echo "Local var dir: $LOCAL_VAR"
echo

# Create local directory structure
echo "Creating local directory structure..."
mkdir -p "$LOCAL_VAR/pmns"
mkdir -p "$LOCAL_VAR/pmdas/darwin"

# Copy darwin PMNS
echo "Copying Darwin PMNS..."
cp "$BUILD_DIR/src/pmdas/darwin/root_darwin" "$LOCAL_VAR/pmns/root"

# Copy help files if they exist
if [ -f "$BUILD_DIR/src/pmdas/darwin/help.dir" ]; then
    echo "Copying Darwin help files..."
    cp "$BUILD_DIR/src/pmdas/darwin/help.dir" "$LOCAL_VAR/pmdas/darwin/"
    cp "$BUILD_DIR/src/pmdas/darwin/help.pag" "$LOCAL_VAR/pmdas/darwin/"
fi

# Create local pcp.conf
echo "Creating local pcp.conf..."
cat > "$SCRIPT_DIR/local-pcp.conf" << EOFCONF
# Local PCP configuration for development
# Generated by setup-local-pcp.sh
# No system installation required!

PCP_ETC_DIR=/private/etc
PCP_SYSCONF_DIR=/private/etc/pcp
PCP_RC_DIR=/private/etc/init.d
PCP_SERVICES_DIR=/usr/local/libexec/pcp/services
PCP_SYSCONFIG_DIR=/private/etc/sysconfig
PCP_BIN_DIR=/usr/local/bin
PCP_BINADM_DIR=/usr/local/libexec/pcp/bin
PCP_LIBADM_DIR=/usr/local/libexec/pcp/lib
PCP_LIB_DIR=/usr/local/lib
PCP_LIB32_DIR=/usr/local/lib
PCP_SHARE_DIR=/usr/local/share/pcp
PCP_INC_DIR=/usr/local/include/pcp
PCP_MAN_DIR=/usr/local/share/man

# LOCAL OVERRIDE: Point to local development directories
PCP_VAR_DIR=$LOCAL_VAR
PCP_PMDAS_DIR=$LOCAL_VAR/pmdas

PCP_PMCDCONF_PATH=/private/etc/pcp/pmcd/pmcd.conf
PCP_PMCDOPTIONS_PATH=/private/etc/pcp/pmcd/pmcd.options
PCP_PMCDRCLOCAL_PATH=/usr/local/libexec/pcp/services/local
PCP_PMPROXYOPTIONS_PATH=/private/etc/pcp/pmproxy/pmproxy.options
PCP_PMIECONTROL_PATH=/private/etc/pcp/pmie/control
PCP_PMSNAPCONTROL_PATH=/private/etc/pcp/pmsnap/control
PCP_PMLOGGERCONTROL_PATH=/private/etc/pcp/pmlogger/control
PCP_PMDASADM_DIR=/usr/local/libexec/pcp/pmdas
PCP_RUN_DIR=/private/var/run/pcp
PCP_LOG_DIR=/private/var/log/pcp
PCP_REMOTE_ARCHIVE_DIR=/private/var/log/pcp/pmproxy
PCP_ARCHIVE_DIR=/private/var/log/pcp/pmlogger
PCP_ARCHIVE_VERSION=3
PCP_SA_DIR=/private/var/log/pcp/sa
PCP_PACCT_SYSTEM_PATH=/private/var/account/pacct
PCP_TMP_DIR=/private/var/lib/pcp/tmp
PCP_TMPFILE_DIR=/private/var/tmp
PCP_DOC_DIR=/usr/local/share/doc/pcp-7.1.0
PCP_HTML_DIR=/usr/local/share/doc/pcp/html
PCP_DEMOS_DIR=/usr/local/share/pcp/demos
PCP_TLSCONF_PATH=/private/etc/pcp/tls.conf
PCP_AWK_PROG="/usr/bin/awk"
PCP_SORT_PROG=/usr/bin/sort
PCP_MAKE_PROG=/usr/bin/make
PCP_CC_PROG="gcc "
PCP_ECHO_PROG=/bin/echo
PCP_ECHO_N=
PCP_ECHO_C=\\c
PCP_SYSLOG_PROG=logger
PCP_PS_PROG=ps
PCP_PS_ALL_FLAGS="-axw -o user,pid,ppid,cpu,stime,tty,time,command"
PCP_PYTHON_PROG=/opt/homebrew/bin/python3
PCP_PERL_PROG=perl
PCP_WHICH_PROG=/usr/bin/which
PCP_PLATFORM=darwin
PCP_PLATFORM_PATHS=/usr/local/bin:/opt/homebrew/bin
PCP_VERSION=7.1.0
PCP_XCONFIRM_PROG=/usr/local/bin/pmconfirm
PCP_USER=pcp
PCP_GROUP=pcp
PCP_SASLCONF_DIR=/private/etc/sasl2
PCP_SYSTEMDUNIT_DIR=
PCP_SYSTEMDSYSUSERS_DIR=/sysusers.d
PCP_SYSTEMDTMPFILES_DIR=/tmpfiles.d
PCP_LOG_RC_SCRIPTS=false
EOFCONF

echo
echo "=== Setup Complete ==="
echo
echo "Local PCP environment created successfully!"
echo
echo "To use for development:"
echo "  cd $SCRIPT_DIR"
echo "  ./dev-test.sh"
echo
echo "Or manually:"
echo "  export PCP_CONF=$SCRIPT_DIR/local-pcp.conf"
echo "  export DYLD_LIBRARY_PATH=$BUILD_DIR/src/libpcp/src:$BUILD_DIR/src/libpcp_pmda/src"
echo "  $BUILD_DIR/src/dbpmda/src/dbpmda"
echo
