/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
*/

// spmatrix.cxx: implementation of the CSPmatrix class.
//
//////////////////////////////////////////////////////////////////////

#include <string.h>
#include "S_math.h"
#include "spmatrix.h"
#include "spexcept.h"
#include "spint.h"
#include "spnum.h"
#include "spcall.h"
#include "spalcfrm.h"
#include "S_y_tab.h"

extern "C" s_object *s_dqrls(s_object *s_x, s_object *s_y, s_object *s_tol);

#ifndef WIN32
#define __min(a,b) ((a)<(b) ? (a) : (b))
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

//////////////////////// CSPmatrix ////////////////////////

//Default constructor
CSPmatrix::CSPmatrix()
: CSParray()
{
}
//Copy constructor 
CSPmatrix::CSPmatrix(const CSPmatrix& sObject)
: CSParray()
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}

//Construct from an object of the base class
CSPmatrix::CSPmatrix(const CSPobject& sObject)
: CSParray()
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}

//Construct from a valid S-expression
CSPmatrix::CSPmatrix(const char* pszExpression)
: CSParray()
{
	CSPevaluator sEvaluator;
	s_object* ps_value = sEvaluator.Eval(pszExpression);
	Attach(sEvaluator.CloneIfNeeded(ps_value), TRUE);
}

//Construct from s_object* 
CSPmatrix::CSPmatrix(s_object* ps_object, BOOL bTryToFreeOnDetach)
: CSParray()
{
	Attach(ps_object, bTryToFreeOnDetach);
}

//Assigment operator
CSPmatrix& CSPmatrix::operator=(const CSPmatrix& sObject)
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
	return *this;
}
//Assigment from the base class
CSPmatrix& CSPmatrix::operator=(const CSPobject& sObject)
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
	return *this;
}
//Assigment operator
CSPmatrix& CSPmatrix::operator=(s_object* ps_object)
{
	Attach(ps_object, FALSE);
	return *this;
}

CSPmatrix::CSPmatrix(long nRow, long nCol, int mode)
: CSParray()
{
	Create(nRow, nCol, mode);
}

CSPmatrix::CSPmatrix(const CSPvector& sVector, long nRow, long nCol)
: CSParray()
{
	Create(sVector, nRow, nCol);
}

CSPmatrix::CSPmatrix(double* pdValues, long nRow, long nCol)
: CSParray()
{
	Create(pdValues, nRow, nCol);
}

CSPmatrix::~CSPmatrix() //The base class CSPobject is responsible for release/detact the object
{
}

//TODO: unitfy the three Create() methods below
BOOL CSPmatrix::Create(long nRow, long nCol, int mode, const char* pszName, long lDataBase)
{
	if ( nRow <=0 || nCol <=0 )
		return FALSE;
	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;

		//Note that new_S_object only allocate the header part.  See copy_with_slots which call value = copy_lev(object, 0, ...)
		s_object* ps_matrix = new_S_object(s_matrix_class, S_evaluator);
		DATA_SLOT(ps_matrix) = sEvaluator.alcvec(mode, nRow*nCol);
	  long i = get_slot_offset(s_array_class, SPL_DOT_DIM, S_evaluator);
	  if ( i < 0 )
	 		SCONNECT_ThrowException("Missing dimensions slot");
			
	  s_object* ps_dim = alcvec(S_MODE_INT, 2, S_evaluator); //.Dim must by an integer vector of length 2
		LIST_POINTER (ps_matrix)[i] = ps_dim;
		INTEGER_POINTER (ps_dim)[0]=nRow;
		INTEGER_POINTER (ps_dim)[1]=nCol;

		//Assign and/or attach
		AttachAndAssign(ps_matrix, pszName, lDataBase);
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	SetTryToFreeOnDetach(TRUE);	
	return IsValid();
}

BOOL CSPmatrix::Create(const CSPvector& sVector, long nRow, long nCol, const char* pszName, long lDataBase)
{
	if( !sVector.IsValid() || nRow <=0 || nCol <=0 || sVector.GetLength() < nRow*nCol 
	||   !SPL_IsAtomicVector(sVector.GetPtr()) 
	)
		return FALSE;
	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;

		//Note that new_S_object only allocate the header part.  See copy_with_slots which call value = copy_lev(object, 0, ...)
		s_object* ps_matrix = new_S_object(s_matrix_class, S_evaluator);
		DATA_SLOT(ps_matrix) = COPY_ALL(sVector.GetPtr()); //copy all rather than sharing!
	  long i = get_slot_offset(s_array_class, SPL_DOT_DIM, S_evaluator);
	  if ( i < 0 )
	 		SCONNECT_ThrowException("Missing dimensions slot");
			
	  s_object* ps_dim = alcvec(S_MODE_INT, 2, S_evaluator); //.Dim must by an integer vector of length 2
		LIST_POINTER (ps_matrix)[i] = ps_dim;
		INTEGER_POINTER (ps_dim)[0]=nRow;
		INTEGER_POINTER (ps_dim)[1]=nCol;

		//Assign and/or attach
		AttachAndAssign(ps_matrix, pszName, lDataBase);

	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	SetTryToFreeOnDetach(TRUE);	
	return IsValid();
}

//It is the caller responsibility to make sure that the length of pdValues is at least nRow*nCol.
//If the length is less than nRow*nCol, random crashs can occur. 
BOOL CSPmatrix::Create(double* pdValues, long nRow, long nCol, const char* pszName, long lDataBase)
{
	if( pdValues == NULL || nRow <=0 || nCol <=0 )
		return FALSE;
	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;

		//Note that new_S_object only allocate the header part.  See copy_with_slots which call value = copy_lev(object, 0, ...)
		s_object* ps_matrix = new_S_object(s_matrix_class, S_evaluator);

		{
			CSPnumeric sNumeric(pdValues, nRow*nCol);
			DATA_SLOT(ps_matrix) = sNumeric.Detach(); //copy all rather than sharing!
		}

	  long i = get_slot_offset(s_array_class, SPL_DOT_DIM, S_evaluator);
	  if ( i < 0 )
	 		SCONNECT_ThrowException("Missing dimensions slot");
			
	  s_object* ps_dim = alcvec(S_MODE_INT, 2, S_evaluator); //.Dim must by an integer vector of length 2
		LIST_POINTER (ps_matrix)[i] = ps_dim;
		INTEGER_POINTER (ps_dim)[0]=nRow;
		INTEGER_POINTER (ps_dim)[1]=nCol;

		//Assign and/or attach
		AttachAndAssign(ps_matrix, pszName, lDataBase);

	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	SetTryToFreeOnDetach(TRUE);	
	return IsValid();
}

//////////////////////////////////////////////////////
// Attributes
//////////////////////////////////////////////////////
BOOL CSPmatrix::Create(
	const char* pszExpression, //==NULL, uses the default constructor, else use the expression to instantiate it.
	const char* pszName,       //object name
	long lDataBase )				//database
{
	BOOL bSuccess = TRUE;
	try
	{
		bSuccess = CSPobject::BaseCreate("matrix", pszExpression, pszName, lDataBase );
	}
	catch(...)
	{
		bSuccess = FALSE;
	}

	return bSuccess;
}

BOOL CSPmatrix::IsValid(void) const
{
	if(CSPobject::IsValid())
		return IS(GetPtr(), s_matrix_class);

	return FALSE;
}

//Coerce() coerces to "matrix" class and attatch
void CSPmatrix::Attach(s_object *ps_object, BOOL bTryToFreeOnDetach)
{
	CSPevaluator sEvaluator;

	if(&(*this) == ps_object)
		return;
	else if(SPL_NotThere(ps_object))
	{
		CSPobject::Attach(NULL); //NULL is OK the same way as the default constructor.
	}
	else if(IS_OBJ(ps_object, matrix))//already SV4 "matrix"
	{
		CSPobject::Attach(ps_object, bTryToFreeOnDetach);
	}
	else if(ps_object->mode == S_MODE_STRUCTURE //potentially a old-style of "matrix", i.e. model.matrix?
	&& ps_object->Class == s_structure_class )
    /* old-style object with "class" attribute; e.g., matrix */
  {
		//Special: work-around for SV3 class "model.matrix" because As(x, "matrix") fails!
		CSPevaluator sEvaluator;
		s_object* ps_new_object = new_S_object(s_matrix_class, S_evaluator);
		DATA_SLOT(ps_new_object) = sEvaluator.CreateNewHeader(DATA_SLOT(ps_object));
		s_object* ps_dim = GET_DIM(ps_object);
		if(!SPL_NotThere(ps_dim))
			LIST_POINTER(ps_new_object)[ARRAY_DIM_SLOT] = sEvaluator.CreateNewHeader(ps_dim);
		else
		{
			CSPinteger sDim(2);
			sDim.SetAt(0, GET_LENGTH(DATA_SLOT(ps_object)));
			sDim.SetAt(1, 1L);
			LIST_POINTER(ps_new_object)[ARRAY_DIM_SLOT] = sDim.Detach();
		}

		s_object* ps_dimNames = GET_DIMNAMES(ps_object);
		if(!SPL_NotThere(ps_dimNames))
			LIST_POINTER(ps_new_object)[ARRAY_DIMNAMES_SLOT] = sEvaluator.CreateNewHeader(ps_dimNames);
		CSPobject::Attach(ps_new_object, bTryToFreeOnDetach);
	}
	else
	{
		CSPobject::BaseCoerceAttach(ps_object, s_matrix_class, bTryToFreeOnDetach);
	}
	return;
}

void CSPmatrix::Validate(void) const
{
	if (!CSPmatrix::IsValid())
		throw SCONNECT_INVALID_SOBJECT;
}
//Return TRUE if lMode is one of valid data mode
BOOL CSPmatrix::IsValidDataMode(long lMode) const
{
	switch(lMode)
	{
	case S_MODE_LGL:
	case S_MODE_INT:
	case S_MODE_REAL:
	case S_MODE_DOUBLE:
	case S_MODE_CHAR:
	case S_MODE_COMPLEX:
	case S_MODE_RAW:
		return TRUE;
		break;
	default:
		break;
	}
	return FALSE;
}

//Return the address to the content of an element, zero-index base
void* CSPmatrix::GetElementPtr(long lRowZeroIndex, long lColZeroIndex, BOOL bValidate) const
{
	if(bValidate)
		Validate();

	void* pReturn=NULL;
	long lDataIndex = lColZeroIndex* GetNRow(FALSE) + lRowZeroIndex;
	s_object* ps_data = GetData(FALSE);
	if ( SPL_NotThere(ps_data) )
		SCONNECT_ThrowException("Missing data slot");

	if( (lDataIndex < 0) || (lDataIndex >= ps_data->length) )
		SCONNECT_ThrowException("Invalid index %d", lDataIndex);

	switch(ps_data->mode) 
	{
		case S_MODE_LGL:
		case S_MODE_INT:
			pReturn = &INTEGER_POINTER(ps_data)[lDataIndex];
			break;
		case S_MODE_REAL:
			pReturn = &SINGLE_POINTER(ps_data)[lDataIndex];
			break;
		case S_MODE_DOUBLE:
			pReturn = &NUMERIC_POINTER(ps_data)[lDataIndex];
			break;
		case S_MODE_CHAR:
			pReturn =	&CHARACTER_POINTER(ps_data)[lDataIndex]; 
			break;
		case S_MODE_COMPLEX:
			pReturn =	&COMPLEX_POINTER(ps_data)[lDataIndex];
			break;
		case S_MODE_RAW:
			pReturn =	&RAW_POINTER(ps_data)[lDataIndex]; 
			break;
		default: 
			SCONNECT_ThrowException("Invalid data mode %d", ps_data->mode);
	}	
	return pReturn;
}

long CSPmatrix::GetNRow(BOOL bValidate) const
{
	s_object* ps_dim = GetDim(bValidate);
	if(bValidate)
	{
		Validate();
		if ( SPL_NotThere(ps_dim) || GET_LENGTH(ps_dim) < 2 || ps_dim->mode!=S_MODE_INT)
			SCONNECT_ThrowException("Invalid dimensions slot");
	}
	return (long) INTEGER_DATA(ps_dim)[0];
}

//Reset number of row "nrow"
BOOL CSPmatrix::SetNRow(long lNRow, BOOL bValidate)
{
	if(bValidate)
		Validate();
	if(lNRow < 0)
		SCONNECT_ThrowException("Invalid number of rows %d", lNRow);

	s_object* ps_dim = GetDim(FALSE);
	if ( SPL_NotThere(ps_dim) || ps_dim->mode!=S_MODE_INT || GET_LENGTH(ps_dim) != 2 )
		SCONNECT_ThrowException("Invalid dimensions slot");

	long nRow, nCol;
	nRow = INTEGER_POINTER(ps_dim)[0];
	nCol = INTEGER_POINTER(ps_dim)[1];

	if ( lNRow == nRow )
		return TRUE;

	BOOL bSuccess = FALSE;

	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;

		//Protect previous header and its immediate arena for the matrix object
		s_object* ps_matrix = sEvaluator.CopyForWrite(GetPtr());

		s_object* ps_data = GetData(FALSE);
		if ( SPL_NotThere(ps_data) )
			SCONNECT_ThrowException("Missing data slot");
		ps_data = sEvaluator.CopyForWrite(ps_data);
		int mode = ps_data->mode;
		long lSize = mode_size(mode);
		s_object* ps_dimnames = GetDimnames(FALSE);
		s_object* ps_rownames = blt_in_NULL;
		if ( !SPL_NotThere(ps_dimnames) && ps_dimnames->mode == S_MODE_LIST && 
			  GET_LENGTH(ps_dimnames)== 2 )
		{
			ps_rownames = LIST_POINTER(ps_dimnames)[0];
		}
		s_object* ps_newRN = ps_rownames;

		long incNew = lSize*lNRow;
		long incOld = lSize*nRow;
		long nBytesCpy = __min(incNew,incOld);
		long nCharCpy  = __min(lNRow,nRow);
		s_object* ps_newData = ::alcvec(mode, nCol*lNRow, S_evaluator);
		char* pvNew = RAW_POINTER(ps_newData);
		char* pvOld = RAW_POINTER(ps_data);
		long i;
		for ( i = 0; i<nCol; i++, pvNew+=incNew, pvOld+=incOld )
		{
			if ( mode == S_MODE_CHAR )
			{
				copy_strings((char**)pvNew, (char**)pvOld, nCharCpy, S_evaluator);
			}
			else
			{
				memcpy(pvNew, pvOld, nBytesCpy);
			}
		}
		// Creating a header for dim and copying data if needed may be overkill?
		ps_dim  = sEvaluator.CopyForWrite(ps_dim);
		INTEGER_POINTER(ps_dim)[0] = lNRow;
		INTEGER_POINTER(ps_dim)[1] = nCol;
		// Ensured i is valid since GetDim() makes a check
		i = get_slot_offset(s_matrix_class, SPL_DOT_DIM, S_evaluator);
		LIST_POINTER(ps_matrix)[i] = ps_dim;

		if ( !SPL_NotThere(ps_rownames) && (nRow=GET_LENGTH(ps_rownames))>0 )
		{
			ps_dimnames  = sEvaluator.CopyForWrite(ps_dimnames);
			ps_newRN  = sEvaluator.CopyForWrite(ps_rownames);
			SET_LENGTH(ps_newRN, lNRow);
			for ( long j=nRow; j<lNRow; j++ )
			{
				CHARACTER_POINTER(ps_newRN)[j] = EMPTY_STRING;
			}
			if(S_is_env_removable_char())
				S_copy_strings(ps_newRN, ps_newRN, S_evaluator);
			LIST_POINTER(ps_dimnames)[0] = ps_newRN;
			i = get_slot_offset(s_matrix_class, SPL_DOT_DIMNAMES, S_evaluator);
			LIST_POINTER(ps_matrix)[i] = ps_dimnames;
		}
		DATA_SLOT(ps_matrix) = ps_newData;
		ReAttachAndAssign(ps_matrix);
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	return bSuccess;
}

//Return number of colums
long CSPmatrix::GetNCol(BOOL bValidate) const
{
	s_object* ps_dim = GetDim(bValidate);
	if(bValidate)
	{
		Validate();
		if ( SPL_NotThere(ps_dim) || GET_LENGTH(ps_dim) < 2 || ps_dim->mode!=S_MODE_INT)
			SCONNECT_ThrowException("Invalid dimensions slot");
	}
	return (long) INTEGER_DATA(ps_dim)[1];
}

//Set/re-set length of "data.frame" vector, number of columns
BOOL CSPmatrix::SetNCol(long lNCol, BOOL bValidate)
{
	s_object* ps_dim = GetDim(bValidate);
	if(bValidate)
	{
		Validate();
		if(lNCol < 0)
			SCONNECT_ThrowException("Invalid number of col: %d", lNCol);
		if ( SPL_NotThere(ps_dim) || ps_dim->mode!=S_MODE_INT || GET_LENGTH(ps_dim) != 2 )
			SCONNECT_ThrowException("Invalid dimensions slot");
	}

	long nRow, nCol;
	nRow = INTEGER_POINTER(ps_dim)[0];
	nCol = INTEGER_POINTER(ps_dim)[1];

	if ( lNCol == nCol )
		return TRUE;

	// unused: BOOL bSuccess = FALSE;

	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;

		//Protect previous header and its immediate arena for the matrix object
		s_object* ps_matrix = sEvaluator.CopyForWrite(GetPtr());

		s_object* ps_data = GetData(FALSE);
		if ( SPL_NotThere(ps_data) )
			SCONNECT_ThrowException("Missing data slot");
		int mode = ps_data->mode;
		long lSize = mode_size(mode);
		s_object* ps_dimnames = GetDimnames(FALSE);
		s_object* ps_colnames = blt_in_NULL;
		if ( !SPL_NotThere(ps_dimnames) && ps_dimnames->mode == S_MODE_LIST && 
			  GET_LENGTH(ps_dimnames)== 2 )
		{
			ps_colnames = LIST_POINTER(ps_dimnames)[1];
		}
		s_object* ps_newCN = ps_colnames;

		long nBytes = lSize*nRow*nCol;
		s_object* ps_newData = alcvec(mode, nRow*lNCol, S_evaluator);
		if ( mode == S_MODE_CHAR )
		{
			copy_strings((char**)RAW_POINTER(ps_newData), (char**)RAW_POINTER(ps_data), nRow*nCol, S_evaluator);
		}
		else
		{
			memcpy(RAW_POINTER(ps_newData), RAW_POINTER(ps_data), nBytes);
		}
		// Creating a header for dim and copying data if needed may be overkill?
		ps_dim  = sEvaluator.CopyForWrite(ps_dim);
		INTEGER_POINTER(ps_dim)[0] = nRow;
		INTEGER_POINTER(ps_dim)[1] = lNCol;
		// Ensured i is valid since GetDim() makes a check
		long i = get_slot_offset(s_matrix_class, SPL_DOT_DIM, S_evaluator);
		LIST_POINTER(ps_matrix)[i] = ps_dim;

		if ( !SPL_NotThere(ps_colnames) && (nCol=GET_LENGTH(ps_colnames))>0 )
		{
			ps_dimnames  = sEvaluator.CopyForWrite(ps_dimnames);
			ps_newCN  = sEvaluator.CopyForWrite(ps_colnames);
			SET_LENGTH(ps_newCN, lNCol);
			for ( i=nCol; i<lNCol; i++ )
				CHARACTER_POINTER(ps_newCN)[i] = EMPTY_STRING;
			S_set_arena_why(ps_newCN, ATOMIC_WHY, S_evaluator);

			LIST_POINTER(ps_dimnames)[1] = ps_newCN;
			long i = get_slot_offset(s_matrix_class, SPL_DOT_DIMNAMES, S_evaluator);
			LIST_POINTER(ps_matrix)[i] = ps_dimnames;
		}
		DATA_SLOT(ps_matrix) = ps_newData;

		ReAttachAndAssign(ps_matrix);
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	return TRUE;
}

CSPcharacter CSPmatrix::GetRowNames(BOOL bValidate) const
{
	if(bValidate)
		Validate();
	s_object* ps_dimnames = GetDimnames(FALSE);
	if ( SPL_NotThere(ps_dimnames) )
		return CSPcharacter();
	if ( ps_dimnames->mode!=S_MODE_LIST || GET_LENGTH(ps_dimnames) < 2 )
		SCONNECT_ThrowException("Invalid dimnames slot");
	s_object* ps_names = LIST_POINTER(ps_dimnames)[0];
	if ( ps_names->mode != S_MODE_CHAR && ps_names->mode!=S_MODE_NULL )
		SCONNECT_ThrowException("Invalid dimnames slot");

	return CSPcharacter(ps_names);
}

CSPcharacter CSPmatrix::GetColNames(BOOL bValidate) const
{
	if(bValidate)
		Validate();
	s_object* ps_dimnames = GetDimnames(FALSE);
	if ( SPL_NotThere(ps_dimnames) )
		return CSPcharacter();
	if ( ps_dimnames->mode!=S_MODE_LIST || GET_LENGTH(ps_dimnames) < 2 )
		SCONNECT_ThrowException("Invalid dimnames slot");
	s_object* ps_names = LIST_POINTER(ps_dimnames)[1];
	if ( ps_names->mode != S_MODE_CHAR && ps_names->mode!=S_MODE_NULL )
		SCONNECT_ThrowException("Invalid dimnames slot");

	return CSPcharacter(ps_names);
}

//Return the memory for the column corresponding to lCol: Zero-based index.
void* CSPmatrix::GetRawColPtrAt(long lCol, BOOL bValidate) const
{
	if(bValidate)
		Validate();
	long nCol = GetNCol(FALSE);
	if ( lCol<0 || lCol>=nCol )
		SCONNECT_ThrowException("Invalid column index %d", lCol);

	s_object* ps_data = GetData(FALSE);
	if ( SPL_NotThere(ps_data) )
		SCONNECT_ThrowException("Missing data slot");

	size_t nBytes = mode_size(GetDataMode(FALSE))*GetNRow(FALSE)*lCol;

	return (void*)(RAW_POINTER(ps_data)+nBytes);
}

void* CSPmatrix::GetRawColPtrAt(const char* pszName, BOOL bValidate) const
{
	if(bValidate)
		Validate();
	if ( !pszName || !*pszName )
		return NULL;
	//Return pointer to the .Dimnames slot
	s_object* ps_names = GetColNames(FALSE);
	for ( long i=0; i<GET_LENGTH(ps_names); i++ )
	{
		if (strcmp(CHARACTER_POINTER(ps_names)[i], pszName) == 0 )
		{
			return GetRawColPtrAt(i, FALSE);
		}
	}
	return NULL;
}

// Get a copy of the vector at column lCol (zero based)
CSPobject CSPmatrix::GetAt(long lCol) const
{
	Validate();
	CSPallocFrame allocFrame;

	void* pvData = GetRawColPtrAt(lCol, FALSE);

	int mode = GetDataMode(FALSE);
	long nRow = GetNRow(FALSE);
	size_t nBytes = mode_size(mode)*nRow;

	s_object* ps_ret = alcvec(mode, nRow, S_evaluator);

	if ( mode == S_MODE_CHAR )
		copy_strings(CHARACTER_POINTER(ps_ret), (char**)pvData, nRow, S_evaluator);
	else
		memcpy(RAW_POINTER(ps_ret), pvData, nBytes);

	return CSPobject(ps_ret, TRUE);
}

// Get a copy of the vector at column with name pazName
CSPobject CSPmatrix::GetAt(const char* pszName) const
{
		Validate();
	if ( !pszName || !*pszName )
		return CSPobject();
	//Return pointer to the .Dimnames slot
	s_object* ps_names = GetColNames(FALSE);
	for ( long i=0; i<GET_LENGTH(ps_names); i++ )
	{
		if (strcmp(CHARACTER_POINTER(ps_names)[i], pszName) == 0 )
		{
			return GetAt(i);
		}
	}
	return CSPobject();
}

static s_class* ClassForMode(int mode)
{
	s_class* ps_class = s_NULL_class;
	switch(mode)
	{
		case S_MODE_LGL:
		{
			ps_class = s_logical_class;
			break;
		}
		case S_MODE_INT:
		{
			ps_class = s_integer_class;
			break;
		}
		case S_MODE_REAL:
		{
			ps_class = s_single_class;
			break;
		}
		case S_MODE_DOUBLE:
		{
			ps_class = s_numeric_class;
			break;
		}
		case S_MODE_COMPLEX:
		{
			ps_class = s_complex_class;
			break;
		}
		case S_MODE_CHAR:
		{
			ps_class = s_character_class;
			break;
		}
		default:
		{
			SCONNECT_ThrowException("Incompatable mode %d", mode);
		}
	}
	return ps_class;
}


//Set the values corrensponding to the column lCol: Zero-based index.
BOOL CSPmatrix::SetAt(s_object* ps_vector, long lCol, const char* pszName, BOOL bValidate)
{
	if(bValidate)
		Validate();
	if ( SPL_NotThere(ps_vector) || !SPL_IsAtomicVector(ps_vector) )
		SCONNECT_ThrowException("Object to be inserted must be atomic of mode integer, real, double, character, or complex");
	if ( lCol < 0 ) 
		SCONNECT_ThrowException("Invalid column index %d", lCol);

	s_object* ps_dim = GetDim(FALSE);
	if ( SPL_NotThere(ps_dim) || ps_dim->mode!=S_MODE_INT || GET_LENGTH(ps_dim) != 2 )
		SCONNECT_ThrowException("Invalid dimensions slot");

	long nRow, nCol;
	nRow = INTEGER_POINTER(ps_dim)[0];
	nCol = INTEGER_POINTER(ps_dim)[1];

	// unused: BOOL bSuccess = FALSE;

	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;

		int mode = 0;
		int mdVec = ps_vector->mode;
		s_object* ps_data = GetData(FALSE);
		s_object* ps_data1 = blt_in_NULL;
		// matrix(NULL) creates a matrix with NULL data
		if ( SPL_NotThere(ps_data) )
		{
			ps_data = blt_in_NULL;
			// Even though the data is null, nCol = 1. It should be zero. 
			nCol = nRow = 0;
			mode = mdVec;
		}
		else
		{
			ps_data1 = ps_data;
			mode = ps_data->mode;
			if ( mdVec < mode )
			{
				s_class* ps_class = ClassForMode(mode);
				ps_vector = AS(ps_vector, ps_class);		
				if ( SPL_NotThere(ps_vector) )
					SCONNECT_ThrowException("Failed to coerce object to class %s", (ps_class?ps_class->name:"????"));

			}
			else if ( mode < mdVec )
			{
				s_class* ps_class = ClassForMode(mdVec);
				ps_data = AS(ps_data, ps_class);		
				if ( SPL_NotThere(ps_data) )
					SCONNECT_ThrowException("Failed to coerce object to class %s", (ps_class?ps_class->name:"????"));

				mode = mdVec;
			}
		}

		//Protect previous header and its immediate arena for the matrix object
		s_object* ps_matrix = NULL;
		long lSize = mode_size(mode);
		long lenVec = GET_LENGTH(ps_vector);
		long nColNew = (lCol>=nCol?lCol+1:nCol);
		long nRowNew = (lenVec>nRow?lenVec:nRow);

		s_object* ps_dimnames = GetDimnames(FALSE);
		s_object* ps_rownames = blt_in_NULL;
		s_object* ps_colnames = blt_in_NULL;
		if ( !SPL_NotThere(ps_dimnames) )
		{
			ps_rownames = LIST_POINTER(ps_dimnames)[0];
			ps_colnames = LIST_POINTER(ps_dimnames)[1];
		}
		s_object* ps_newRN = ps_rownames;
		s_object* ps_newCN = ps_colnames;

		BOOL bCopyVector = TRUE;
		if ( nColNew != nCol || nRowNew != nRow )
		{
			ps_matrix = sEvaluator.CopyForWrite(GetPtr());
			if ( SPL_NotThere(ps_data) && nColNew == 1 )
			{
				// data was NULL. Share the vector.
				ps_data = SPL_NewHeaderForInsertion(ps_vector);
				bCopyVector = FALSE;
			}
			else
			{
				long incNew = lSize*nRowNew;
				long incOld = lSize*nRow;
				s_object* ps_newData = alcvec(mode, nColNew*nRowNew, S_evaluator);
				char* pvNew = RAW_POINTER(ps_newData);
				char* pvOld = RAW_POINTER(ps_data);
				// Could be matrix constructed with no rows (matrix(NULL)) and we 
				//  are setting its first vector
				if ( nRow > 0 )
				{
					for ( long i = 0; i<nCol; i++, pvNew+=incNew, pvOld+=incOld )
					{
						if ( i == lCol )
							continue;
						if ( mode == S_MODE_CHAR )
						{
							copy_strings((char**)pvNew, (char**)pvOld, nRow, S_evaluator);
						}
						else
						{
							memcpy(pvNew, pvOld, incOld);
						}
					}
				}
				ps_data = ps_newData;
			}
			// Creating a header for dim and copying data if needed may be overkill?
			ps_dim  = sEvaluator.CopyForWrite(ps_dim);
			INTEGER_POINTER(ps_dim)[0] = nRowNew;
			INTEGER_POINTER(ps_dim)[1] = nColNew;
			// Ensured i is valid since GetDim() makes a check
			long i = get_slot_offset(s_matrix_class, SPL_DOT_DIM, S_evaluator);
			LIST_POINTER(ps_matrix)[i] = ps_dim;

			if ( nRowNew != nRow && !SPL_NotThere(ps_rownames) )
			{
				ps_newRN  = sEvaluator.CopyForWrite(ps_rownames);
				SET_LENGTH(ps_newRN, nRowNew);
				for ( i=nRow; i<nRowNew; i++ )
				{
					CHARACTER_POINTER(ps_newRN)[i] = EMPTY_STRING;
				}
				if(S_is_env_removable_char())
					S_copy_strings(ps_newRN, ps_newRN, S_evaluator);
			}
		}
		else if ( ps_data1 == ps_data )
		{
			// Did not coerce the mode above, get a new header and copy the data if needed
			ps_data = sEvaluator.CopyForWrite(ps_data);
			ps_matrix = sEvaluator.CopyForWrite(GetPtr());
		}
		else
		{
			ps_matrix = sEvaluator.CopyForWrite(GetPtr());
		}

		if ( bCopyVector )
		{
			long inc = lCol*lSize*nRowNew;
			if ( mode == S_MODE_CHAR )
			{
				copy_strings((char**)(RAW_POINTER(ps_data)+inc), (char**)RAW_POINTER(ps_vector), nRowNew, S_evaluator);
			}
			else
			{
				memcpy(RAW_POINTER(ps_data)+inc,RAW_POINTER(ps_vector),nRowNew*lSize); 
			}
		}

		if ( pszName && *pszName || (nCol!=nColNew && !SPL_NotThere(ps_colnames) && GET_LENGTH(ps_colnames)>0) )
		{
			if ( SPL_NotThere(ps_colnames) )
			{
				ps_newCN = NEW_CHARACTER(nColNew);
				for ( long i=0; i<nColNew; i++ )
				{
					if ( i == lCol )
						continue;
					CHARACTER_POINTER(ps_newCN)[i] = EMPTY_STRING;
				}
			}
			else 
			{	
				ps_newCN  = sEvaluator.CopyForWrite(ps_colnames);
				if ( nColNew != GET_LENGTH(ps_newCN) )
					SET_LENGTH(ps_newCN, nColNew);
			}
			if ( (!pszName || *pszName=='\0') )
				CHARACTER_POINTER(ps_newCN)[lCol] = EMPTY_STRING;
			else
				CHARACTER_POINTER(ps_newCN)[lCol] = SPL_AllocateStringInFrame(ps_newCN, pszName);

			if(S_is_env_removable_char())
				S_copy_strings(ps_newCN, ps_newCN, S_evaluator);
		}
		if ( ps_newCN!=ps_colnames || ps_newRN!=ps_rownames ) 
		{
			if ( SPL_NotThere(ps_dimnames) )
				ps_dimnames = NEW_LIST(2);
			else
				ps_dimnames = sEvaluator.CopyForWrite(ps_dimnames);

			LIST_POINTER(ps_dimnames)[0] = ps_newRN;
			LIST_POINTER(ps_dimnames)[1] = ps_newCN;
			long i = get_slot_offset(s_matrix_class, SPL_DOT_DIMNAMES, S_evaluator);
			LIST_POINTER(ps_matrix)[i] = ps_dimnames;
		}
		DATA_SLOT(ps_matrix) = ps_data;

		ReAttachAndAssign(ps_matrix);
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	return TRUE;
}

// Insert a vector, "data.frame", or matrix at index: zero-base
BOOL CSPmatrix::InsertAt(s_object* ps_vector, long lCol, const char* pszName, BOOL bValidate)
{
	if(bValidate)
		Validate();
	if ( SPL_NotThere(ps_vector) || !SPL_IsAtomicVector(ps_vector) )
		SCONNECT_ThrowException("Object to be inserted must be atomic of mode integer, real, double, character, or complex");
	if(lCol<0)
			SCONNECT_ThrowException("Invalid column index %d", lCol);

	BOOL bSuccess = FALSE;
	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;
		if(GetPtr() == NULL)
			Attach(	new_S_object(s_matrix_class, S_evaluator)); //Create a 1x1 matrix

		s_object* ps_dim = GetDim(FALSE);
		if ( SPL_NotThere(ps_dim) || ps_dim->mode!=S_MODE_INT || GET_LENGTH(ps_dim) != 2 )
			SCONNECT_ThrowException("Invalid dimensions slot");

		long nRow, nCol;
		nRow = INTEGER_POINTER(ps_dim)[0];
		nCol = INTEGER_POINTER(ps_dim)[1];

		int mode = 0;
		int mdVec = ps_vector->mode;
		s_object* ps_data = GetData(FALSE);
		s_object* ps_data1 = blt_in_NULL;
		// matrix(NULL) creates a matrix with NULL data
		if ( SPL_NotThere(ps_data) )
		{
			ps_data = blt_in_NULL;
			// Even though the data is null, nCol = 1. It should be zero. 
			nCol = nRow = 0;
			mode = mdVec;
		}
		else
		{
			ps_data1 = ps_data;
			mode = ps_data->mode;

			if ( mdVec < mode )
			{
				s_class* ps_class = ClassForMode(mode);
				ps_vector = AS(ps_vector, ps_class);		
				if ( SPL_NotThere(ps_vector) )
					SCONNECT_ThrowException("Failed to coerce object to class %s", (ps_class?ps_class->name:"????"));
			}
			else if ( mode < mdVec )
			{
				s_class* ps_class = ClassForMode(mdVec);
				ps_data = AS(ps_data, ps_class);		
				if ( SPL_NotThere(ps_data) )
					SCONNECT_ThrowException("Failed to coerce object to class %s", (ps_class?ps_class->name:"????"));

				mode = mdVec;
			}
		}
		//Protect previous header and its immediate arena for the matrix object
		s_object* ps_matrix = sEvaluator.CopyForWrite(GetPtr());
		long lSize = mode_size(mode);
		long lenVec = GET_LENGTH(ps_vector);
		long nColNew = (lCol>=nCol?lCol+1:nCol+1);
		long nRowNew = (lenVec>nRow?lenVec:nRow);

		s_object* ps_dimnames = GetDimnames(FALSE);
		s_object* ps_rownames = blt_in_NULL;
		s_object* ps_colnames = blt_in_NULL;
		if ( !SPL_NotThere(ps_dimnames) && ps_dimnames->mode == S_MODE_LIST && 
			  GET_LENGTH(ps_dimnames)== 2 )
		{
			ps_rownames = LIST_POINTER(ps_dimnames)[0];
			ps_colnames = LIST_POINTER(ps_dimnames)[1];
		}
		s_object* ps_newRN = ps_rownames;
		s_object* ps_newCN = ps_colnames;

		if ( SPL_NotThere(ps_data) && nColNew == 1 )
		{
			// data was NULL. Share the vector.
			DATA_SLOT(ps_matrix) = SPL_NewHeaderForInsertion(ps_vector);
		}
		else
		{
			long incNew = lSize*nRowNew;
			long incOld = lSize*nRow;
			s_object* ps_newData = alcvec(mode, nColNew*nRowNew, S_evaluator);
			char* pvNew = RAW_POINTER(ps_newData);
			char* pvOld = RAW_POINTER(ps_data);
			if ( nRow > 0 )
			{
				for ( long i = 0; i<__min(lCol,nCol); i++, pvNew+=incNew, pvOld+=incOld )
				{
					if ( mode == S_MODE_CHAR )
					{
						copy_strings((char**)pvNew, (char**)pvOld, nRow, S_evaluator);
					}
					else
					{
						memcpy(pvNew, pvOld, incOld);
					}
				}
			}
			else
			{
				pvNew += lCol*nRowNew*lSize;
			}
			if ( mode == S_MODE_CHAR )
			{
				copy_strings((char**)pvNew, CHARACTER_POINTER(ps_vector), lenVec, S_evaluator);
			}
			else
			{
				memcpy(pvNew, RAW_POINTER(ps_vector), lenVec*lSize);
			}
			if ( nRow > 0 )
			{
				pvNew+=incNew;
				for ( long i = lCol+1; i<nColNew; i++, pvNew+=incNew, pvOld+=incOld )
				{
					if ( mode == S_MODE_CHAR )
					{
						copy_strings((char**)pvNew, (char**)pvOld, nRow, S_evaluator);
					}
					else
					{
						memcpy(pvNew, pvOld, incOld);
					}
				}
			}
			DATA_SLOT(ps_matrix) = ps_newData;
		}

		// Creating a header for dim and copying data if needed may be overkill?
		ps_dim  = sEvaluator.CopyForWrite(ps_dim);
		INTEGER_POINTER(ps_dim)[0] = nRowNew;
		INTEGER_POINTER(ps_dim)[1] = nColNew;
		// Ensured i is valid since GetDim() makes a check
		long i = get_slot_offset(s_matrix_class, SPL_DOT_DIM, S_evaluator);
		LIST_POINTER(ps_matrix)[i] = ps_dim;

		if ( nRowNew != nRow && (!SPL_NotThere(ps_rownames) && GET_LENGTH(ps_rownames)>0) )
		{
			ps_newRN  = sEvaluator.CopyForWrite(ps_rownames);
			SET_LENGTH(ps_newRN, nRowNew);
			for ( i=nRow; i<nRowNew; i++ )
			{
				CHARACTER_POINTER(ps_newRN)[i] = EMPTY_STRING;
			}
		}
		if ( pszName && *pszName || (!SPL_NotThere(ps_colnames) && GET_LENGTH(ps_colnames)>0) )
		{
			if ( SPL_NotThere(ps_colnames) )
			{
				ps_newCN = NEW_CHARACTER(nColNew);
				for ( int i=0; i<nColNew; i++ )
				{
					if ( i == lCol )
						continue;
					CHARACTER_POINTER(ps_newCN)[i] = EMPTY_STRING;
				}
			}
			else 
			{	
				ps_newCN  = sEvaluator.CopyForWrite(ps_colnames);
				SET_LENGTH(ps_newCN, nColNew);
				for ( i=nColNew-1; i>lCol; i-- )
				{
					CHARACTER_POINTER(ps_newCN)[i] = CHARACTER_POINTER(ps_newCN)[i-1];
				}
			}
			if ( (!pszName || *pszName=='\0') )
				CHARACTER_POINTER(ps_newCN)[lCol] = EMPTY_STRING;
			else
				CHARACTER_POINTER(ps_newCN)[lCol] = SPL_AllocateStringInFrame(ps_newCN, pszName);
		}
		if ( ps_newCN!=ps_colnames || ps_newRN!=ps_rownames ) 
		{
			if ( SPL_NotThere(ps_dimnames) )
				ps_dimnames = NEW_LIST(2);
			else
				ps_dimnames = sEvaluator.CopyForWrite(ps_dimnames);

			LIST_POINTER(ps_dimnames)[0] = ps_newRN;
			LIST_POINTER(ps_dimnames)[1] = ps_newCN;
			i = get_slot_offset(s_matrix_class, SPL_DOT_DIMNAMES, S_evaluator);
			LIST_POINTER(ps_matrix)[i] = ps_dimnames;
		}

		ReAttachAndAssign(ps_matrix);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	return bSuccess;
}

// Insert an object created by an expression at index: zero-base
BOOL CSPmatrix::InsertAt(const char* pszExpression, long lCol, const char* pszName, BOOL bValidate)
{
	if(bValidate)
		Validate();
	if(lCol<0)
		SCONNECT_ThrowException("Invalid column index %d", lCol);
	if ( !pszExpression || !*pszExpression )
		SCONNECT_ThrowException("Invalid expression");

	BOOL bSuccess = FALSE;

	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;
		if(GetPtr() == NULL)
			Attach(	new_S_object(s_matrix_class, S_evaluator)); //Create a 1x1 matrix		
		s_object* ps_vector = SPL_DirectParseEval(pszExpression);
		if ( SPL_NotThere(ps_vector) )
			SCONNECT_ThrowException("Failed to evaluate expression: %s",pszExpression);
		bSuccess = InsertAt(ps_vector, lCol, pszName, FALSE);
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}

	return bSuccess;
}

BOOL CSPmatrix::RemoveAllCols(BOOL bValidate)
{
	if(bValidate)
		Validate();
	long nCols = ncol();
	if ( nCols <= 0 )
		return TRUE;
	RemoveAt(0, nCols, FALSE);
	return TRUE;
}

BOOL CSPmatrix::RemoveAt(long lCol, long nCols, BOOL bValidate)
{
	if(bValidate)
		Validate();
	//Get the .Data slot which should be a list object
	s_object* ps_data = GetData(FALSE);
	if( SPL_NotThere(ps_data) )
		SCONNECT_ThrowException("Missing data slot");

	s_object* ps_dim = GetDim(FALSE);
	if ( SPL_NotThere(ps_dim) || ps_dim->mode!=S_MODE_INT || GET_LENGTH(ps_dim) != 2 )
		SCONNECT_ThrowException("Invalid dimensions slot");

	long nRow, nCol;
	nRow = INTEGER_POINTER(ps_dim)[0];
	nCol = INTEGER_POINTER(ps_dim)[1];

	if ( lCol < 0 || lCol >= nCol )
		SCONNECT_ThrowException("Invalid column index %d", lCol);

	BOOL bSuccess = FALSE;

	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;

		int mode = ps_data->mode;

		//Protect previous header and its immediate arena for the matrix object
		s_object* ps_matrix = sEvaluator.CopyForWrite(GetPtr());
		long lSize = mode_size(mode);
		long nColNew = nCol-nCols;

		s_object* ps_dimnames = GetDimnames(FALSE);
		s_object* ps_colnames = blt_in_NULL;
		if ( !SPL_NotThere(ps_dimnames) && ps_dimnames->mode == S_MODE_LIST && 
			  GET_LENGTH(ps_dimnames)== 2 )
		{
			ps_colnames = LIST_POINTER(ps_dimnames)[1];
		}
		s_object* ps_newCN = ps_colnames;

		long incOld = lSize*nRow*(lCol+nCols);
		long nElements[2] = {nRow*lCol,nRow*(nCol-lCol-nCols)};
		s_object* ps_newData = ::alcvec(mode, nColNew*nRow, S_evaluator);
		char* pvNew = RAW_POINTER(ps_newData);
		char* pvOld = RAW_POINTER(ps_data);
		long i;
		for ( i = 0; i<2; pvNew+=nElements[i]*lSize, pvOld+=incOld, i++ )
		{
			if ( mode == S_MODE_CHAR )
			{
				copy_strings((char**)pvNew, (char**)pvOld, nElements[i], S_evaluator);
			}
			else
			{
				memcpy(pvNew, pvOld, nElements[i]*lSize);
			}
		}
		DATA_SLOT(ps_matrix) = ps_newData;
		// Creating a header for dim and copying data if needed may be overkill?
		ps_dim  = sEvaluator.CopyForWrite(ps_dim);
		INTEGER_POINTER(ps_dim)[1] = nColNew;
		// Ensured i is valid since GetDim() makes a check
		i = get_slot_offset(s_matrix_class, SPL_DOT_DIM, S_evaluator);
		LIST_POINTER(ps_matrix)[i] = ps_dim;

		if ( !SPL_NotThere(ps_colnames) && GET_LENGTH(ps_colnames) > 0 )
		{
			ps_newCN = NEW_CHARACTER(nColNew);
			for ( i=0; i<lCol; i++ )
			{
				if ( strlen(CHARACTER_POINTER(ps_colnames)[i])>0 )
					CHARACTER_POINTER(ps_newCN)[i] = SPL_AllocateStringInFrame(ps_newCN, CHARACTER_POINTER(ps_colnames)[i]);
				else
					CHARACTER_POINTER(ps_newCN)[i] = EMPTY_STRING;
			}
			for ( i=lCol+nCols; i<nCol; i++ )
			{
				if ( strlen(CHARACTER_POINTER(ps_colnames)[i])>0 )
					CHARACTER_POINTER(ps_newCN)[i-nCols] = SPL_AllocateStringInFrame(ps_newCN, CHARACTER_POINTER(ps_colnames)[i]);
				else
					CHARACTER_POINTER(ps_newCN)[i-nCols] = EMPTY_STRING;
			}
		}
		if ( ps_newCN!=ps_colnames ) 
		{
			ps_dimnames = sEvaluator.CopyForWrite(ps_dimnames);
			LIST_POINTER(ps_dimnames)[1] = ps_newCN;
			i = get_slot_offset(s_matrix_class, SPL_DOT_DIMNAMES, S_evaluator);
			LIST_POINTER(ps_matrix)[i] = ps_dimnames;
		}

		ReAttachAndAssign(ps_matrix);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}

	return bSuccess;
}

BOOL CSPmatrix::RemoveAt(const char* pszName, BOOL bValidate)
{
	if(bValidate)
		Validate();
	if ( !pszName || !*pszName )
		SCONNECT_ThrowException("Invalid name");

	s_object* ps_names = GetColNames(FALSE);
	if( SPL_NotThere(ps_names) || GET_LENGTH(ps_names) == 0 )
	{
		SCONNECT_Warning("Object has no column names");
		return FALSE;
	}

	if ( !IS_CHARACTER(ps_names) )
		SCONNECT_ThrowException("Invalid column names");

	long numNames = GET_LENGTH(ps_names);
	long numElements = GetNCol(FALSE);
	if ( numNames != numElements )
		SCONNECT_ThrowException("Invalid column names vector");

	for ( long i=0; i<numNames; i++ )
	{
		char* pszNameFound = CHARACTER_POINTER(ps_names)[i];
		if ( strcmp( pszNameFound, pszName ) == 0 )
			return RemoveAt(i, 1, FALSE);
	}
	SCONNECT_ThrowException("Column name \"%s\" not found", pszName);
	return FALSE;
}

// Insert rows at index: zero-base
BOOL CSPmatrix::InsertRows(long lRow, const char* pszName, long nRows, BOOL bValidate)
{
	if(bValidate)
		Validate();
	if(GetPtr() == NULL)
		Attach(	new_S_object(s_matrix_class, S_evaluator));
	if(lRow<0)
		SCONNECT_ThrowException("Invalid row index %d", lRow);

	//Get the .Data slot which should be a list object
	s_object* ps_data = GetData(FALSE);
	if( SPL_NotThere(ps_data) )
		SCONNECT_ThrowException("Missing data slot");

	s_object* ps_dim = GetDim(FALSE);
	if ( SPL_NotThere(ps_dim) || ps_dim->mode!=S_MODE_INT || GET_LENGTH(ps_dim) != 2 )
		SCONNECT_ThrowException("Invalid dimensions slot");

	long nRow, nCol;
	nRow = INTEGER_POINTER(ps_dim)[0];
	nCol = INTEGER_POINTER(ps_dim)[1];
	if ( lRow > nRow )
	{
		nRows = lRow-nRow+nRows;
		lRow = nRow;
	}

	BOOL bSuccess = FALSE;

	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;

		int mode = ps_data->mode;

		//Protect previous header and its immediate arena for the matrix object
		s_object* ps_matrix = sEvaluator.CopyForWrite(GetPtr());
		long lSize = mode_size(mode);
		long nRowNew = nRows+nRow;

		s_object* ps_dimnames = GetDimnames(FALSE);
		s_object* ps_rownames = blt_in_NULL;
		if ( !SPL_NotThere(ps_dimnames) && ps_dimnames->mode == S_MODE_LIST && 
			  GET_LENGTH(ps_dimnames)== 2 )
		{
			ps_rownames = LIST_POINTER(ps_dimnames)[0];
		}
		s_object* ps_newRN = ps_rownames;

		long incNew[2] = {lSize*(lRow+nRows), lSize*(nRowNew-lRow-nRows)};
		long incOld;
		long nElements[2] = {lRow,nRow-lRow};
		s_object* ps_newData = alcvec(mode, nCol*nRowNew, S_evaluator);
		char* pvNew = RAW_POINTER(ps_newData);
		char* pvOld = RAW_POINTER(ps_data);
		long i;
		for ( i = 0; i<nCol; i++)
		{
			for ( int j=0; j<2; pvNew += incNew[j], j++ )
			{
				incOld = nElements[j]*lSize;
				if ( mode == S_MODE_CHAR )
				{
					copy_strings((char**)pvNew, (char**)pvOld, nElements[j], S_evaluator);
				}
				else if ( incOld )
				{
					memcpy(pvNew, pvOld, incOld);
				}			
				pvOld += incOld;
			}
		}
		DATA_SLOT(ps_matrix) = ps_newData;
		// Creating a header for dim and copying data if needed may be overkill?
		ps_dim  = sEvaluator.CopyForWrite(ps_dim);
		INTEGER_POINTER(ps_dim)[0] = nRowNew;
		// Ensured i is valid since GetDim() makes a check
		i = get_slot_offset(s_matrix_class, SPL_DOT_DIM, S_evaluator);
		LIST_POINTER(ps_matrix)[i] = ps_dim;

		if ( (!SPL_NotThere(ps_rownames) && GET_LENGTH(ps_rownames)>0) || (pszName && *pszName) )
		{
			if ( SPL_NotThere(ps_rownames) )
			{
				ps_newRN = NEW_CHARACTER(nRowNew);
			}
			else
			{
				ps_newRN  = sEvaluator.CopyForWrite(ps_rownames);
				SET_LENGTH(ps_newRN, nRowNew);
				for ( i=1; i<=nRow-lRow; i++ )
					CHARACTER_POINTER(ps_newRN)[nRowNew-i] = CHARACTER_POINTER(ps_newRN)[nRow-i];
			}

			for ( i=lRow; i<lRow+nRows; i++ )
			{
				if ( (!pszName || *pszName=='\0') )
					CHARACTER_POINTER(ps_newRN)[i] = EMPTY_STRING;
				else
					CHARACTER_POINTER(ps_newRN)[i] = SPL_AllocateStringInFrame(ps_newRN, pszName);
			}
		}
		if ( ps_newRN!=ps_rownames ) 
		{
			ps_dimnames = sEvaluator.CopyForWrite(ps_dimnames);
			LIST_POINTER(ps_dimnames)[0] = ps_newRN;
			i = get_slot_offset(s_matrix_class, SPL_DOT_DIMNAMES, S_evaluator);
			LIST_POINTER(ps_matrix)[i] = ps_dimnames;
		}

		ReAttachAndAssign(ps_matrix);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}

	return bSuccess;
}

BOOL CSPmatrix::RemoveRows(long lRow, long nRows, BOOL bValidate)
{
	if(bValidate)
		Validate();
	if ( nRows == 0 )
		return TRUE;
	//Get the .Data slot which should be a list object
	s_object* ps_data = GetData(FALSE);
	if( SPL_NotThere(ps_data) )
		SCONNECT_ThrowException("Missing data slot");

	s_object* ps_dim = GetDim(FALSE);
	if ( SPL_NotThere(ps_dim) || ps_dim->mode!=S_MODE_INT || GET_LENGTH(ps_dim) != 2 )
		SCONNECT_ThrowException("Invalid dimensions slot");

	long nRow, nCol;
	nRow = INTEGER_POINTER(ps_dim)[0];
	nCol = INTEGER_POINTER(ps_dim)[1];
	if ( lRow<0 || lRow>=nRow )
		SCONNECT_ThrowException("Invalid row index %d", lRow);

	if ( nRows < 0 || lRow+(nRows-1)>=nRow ) 
		SCONNECT_ThrowException("Invalid number of rows %d", nRows);

	BOOL bSuccess = FALSE;

	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;

		int mode = ps_data->mode;

		//Protect previous header and its immediate arena for the matrix object
		s_object* ps_matrix = sEvaluator.CopyForWrite(GetPtr());
		long lSize = mode_size(mode);
		long nRowNew = nRow-nRows;

		s_object* ps_dimnames = GetDimnames(FALSE);
		s_object* ps_rownames = blt_in_NULL;
		if ( !SPL_NotThere(ps_dimnames) && ps_dimnames->mode == S_MODE_LIST && 
			  GET_LENGTH(ps_dimnames)== 2 )
		{
			ps_rownames = LIST_POINTER(ps_dimnames)[0];
		}
		s_object* ps_newRN = ps_rownames;

		long incOld[2] = {lSize*(lRow+nRows), lSize*(nRow-lRow-nRows)};
		long incNew = 0;
		long nElements[2] = {lRow,nRow-lRow-nRows};
		s_object* ps_newData = alcvec(mode, nCol*nRowNew, S_evaluator);
		char* pvNew = RAW_POINTER(ps_newData);
		char* pvOld = RAW_POINTER(ps_data);
		long i;
		for ( i = 0; i<nCol; i++)
		{
			for ( int j=0; j<2; pvOld += incOld[j], pvNew += incNew, j++ )
			{
				incNew = nElements[j]*lSize;
				if ( mode == S_MODE_CHAR )
				{
					copy_strings((char**)pvNew, (char**)pvOld, nElements[j], S_evaluator);
				}
				else if ( incOld )
				{
					memcpy(pvNew, pvOld, incNew);
				}			
			}
		}
		DATA_SLOT(ps_matrix) = ps_newData;
		// Creating a header for dim and copying data if needed may be overkill?
		ps_dim  = sEvaluator.CopyForWrite(ps_dim);
		INTEGER_POINTER(ps_dim)[0] = nRowNew;
		// Ensured i is valid since GetDim() makes a check
		i = get_slot_offset(s_matrix_class, SPL_DOT_DIM, S_evaluator);
		LIST_POINTER(ps_matrix)[i] = ps_dim;

		if ( !SPL_NotThere(ps_rownames) && GET_LENGTH(ps_rownames)>0 )
		{
			ps_newRN  = sEvaluator.CopyForWrite(ps_rownames);
			for ( i=lRow; i<nRowNew; i++ )
			{
				CHARACTER_POINTER(ps_newRN)[i] = CHARACTER_POINTER(ps_newRN)[i+nRows];
			}
			SET_LENGTH(ps_newRN, nRowNew);
		}
		if ( ps_newRN!=ps_rownames ) 
		{
			ps_dimnames = sEvaluator.CopyForWrite(ps_dimnames);
			LIST_POINTER(ps_dimnames)[0] = ps_newRN;
			i = get_slot_offset(s_matrix_class, SPL_DOT_DIMNAMES, S_evaluator);
			LIST_POINTER(ps_matrix)[i] = ps_dimnames;
		}

		ReAttachAndAssign(ps_matrix);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	return bSuccess;
}

BOOL CSPmatrix::SetRowNames(s_object* ps_rownames, BOOL bValidate)
{
	if(bValidate)
		Validate();
	BOOL bIsNull = IS_OBJ(ps_rownames,NULL);
	if ( !bIsNull && !IS_CHARACTER(ps_rownames) )
		SCONNECT_ThrowException("Row names must be character");

	long nRow = GetNRow(FALSE);
	if ( !bIsNull && GET_LENGTH(ps_rownames) != nRow )
		SCONNECT_ThrowException("Row names have an invalid length %d", GET_LENGTH(ps_rownames));

	BOOL bSuccess = FALSE;

	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;

		s_object* ps_dimnames = GetDimnames(FALSE);
		if ( SPL_NotThere(ps_dimnames) )
			ps_dimnames = NEW_LIST(2);
		else 
		{
			ps_dimnames = sEvaluator.CopyForWrite(ps_dimnames);
			if ( GET_LENGTH(ps_dimnames) != 2 )
				SET_LENGTH(ps_dimnames,2);
		}
		s_object* ps_matrix = sEvaluator.CopyForWrite(GetPtr());
		
		LIST_POINTER(ps_dimnames)[0] = (bIsNull? blt_in_NULL: sEvaluator.CreateNewHeader(ps_rownames));
		long i = get_slot_offset(s_matrix_class, SPL_DOT_DIMNAMES, S_evaluator);
		LIST_POINTER(ps_matrix)[i] = ps_dimnames;

		ReAttachAndAssign(ps_matrix);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
		bSuccess = FALSE;
	}

	return bSuccess;
}

BOOL CSPmatrix::SetColNames(s_object* ps_colnames, BOOL bValidate)
{
	if(bValidate)
		Validate();
	BOOL bIsNull = IS_OBJ(ps_colnames,NULL);
	if ( !bIsNull && !IS_CHARACTER(ps_colnames) )
		SCONNECT_ThrowException("Row names must be character");

	long nCol = GetNCol(FALSE);
	if ( !bIsNull && GET_LENGTH(ps_colnames) != nCol )
		SCONNECT_ThrowException("Column names have an invalid length %d", GET_LENGTH(ps_colnames));

	BOOL bSuccess = FALSE;

	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;

		s_object* ps_dimnames = GetDimnames(FALSE);
		if ( SPL_NotThere(ps_dimnames) )
			ps_dimnames = NEW_LIST(2);
		else 
		{
			ps_dimnames = sEvaluator.CopyForWrite(ps_dimnames);
			if ( GET_LENGTH(ps_dimnames) != 2 )
				SET_LENGTH(ps_dimnames,2);
		}
		s_object* ps_matrix = sEvaluator.CopyForWrite(GetPtr());
		
		LIST_POINTER(ps_dimnames)[1] = (bIsNull? blt_in_NULL: sEvaluator.CreateNewHeader(ps_colnames));

		long i = get_slot_offset(s_matrix_class, SPL_DOT_DIMNAMES, S_evaluator);
		LIST_POINTER(ps_matrix)[i] = ps_dimnames;

		ReAttachAndAssign(ps_matrix);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
		bSuccess = FALSE;
	}

	return bSuccess;
}

/*
* ConditionNumber() calls DGECON(LAPACK) to estimate the reciprocal of the condition number of a general
* real matrix A, in either the 1-norm or the infinity-norm, using
* the LU factorization computed by DGETRF.
*
* The condition number is computed as
*      ( norm(A) * norm(inv(A)) ).
*/

double CSPmatrix::ConditionNumber(BOOL bValidate) const
{
	if(bValidate)
		Validate();
	double dReturn= 0.0;

	CSPmatrix sdA(GetPtr());
	if(sdA.GetDataMode(FALSE) != S_MODE_DOUBLE)
		sdA.SetDataMode(S_MODE_DOUBLE);

	long n = sdA.GetNRow(FALSE); //The order of the matrix pdA
	long* pn = &n;
	double dANORM;
	double* pdANORM= &dANORM;
	double dRCOND;

	double* pdWORK = new double[n*4];
	long* pnIWORK = new long[n];
	long nINFO;

	try
	{
		//Get pointer to double
		double* pdA = (double*) sdA.GetElementPtr(0L, 0L, FALSE);

		//Compute 1-norm of A
		dANORM = F77_CALL(dlange)( F77_CHAR_CALL((char*)"1"), pn, pn, pdA, pn, pdWORK);

		//Call the LAPACK routine:DGECON
		F77_CALL(dgecon)( F77_CHAR_CALL((char*)"1"), pn, pdA, pn, pdANORM, &dRCOND, pdWORK, pnIWORK, &nINFO);

		if(nINFO == 0) //suceess?
		  dReturn = 1.0/dRCOND;
		else
			SCONNECT_ThrowException("LAPACK routine DGECON failed");
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}

	delete [] pdWORK;
	delete [] pnIWORK;

	return dReturn;
}

BOOL CSPmatrix::Transpose(BOOL bValidate)
{
	if(bValidate)
		Validate();
	BOOL bSuccess = FALSE;

	try
	{
		//Make sure that top-level-eval is open so that default allocated frame is valid to support temp. obj.
		CSPevaluator sEvaluator;

		CSPcall spt("t");
		s_object* ps_matrix = spt.Eval(GetPtr());

		ReAttachAndAssign(ps_matrix);
		bSuccess = TRUE;
	}
	catch(CSPexception& except)
	{
		except.Print();
	}
	catch(...)
	{
	}
	return bSuccess;

}
//matrix-matrix multiply: %*% 
const CSPmatrix CSPmatrix::Multiply(const CSPmatrix& sRhs) const
{
	return CSPmatrix(SPL_Multiply(*this, sRhs));
}
//matrix-vector multiply: %*% 
const CSPobject CSPmatrix::Multiply(const CSPnumeric& sRhs) const
{
	return CSPobject(SPL_Multiply(*this, sRhs));
}
//QR
CSPmatrix CSPmatrix::QRLS(s_object* ps_y, double dTol, BOOL bValidate) const
{
	if( bValidate )
		Validate();

	if( !S_evaluator->_eval_open )
		SCONNECT_ThrowException(SCONNECT_EVALUATOR_CLOSED);

	if ( GET_DATA(GetPtr())->mode != S_MODE_DOUBLE )
		SCONNECT_ThrowException("QRLS method only valid for matrices of mode double");

	static const char* szNames[] = {"qr", "pivot", "qraux", "coef", "residuals", "qt", "tol", "rank"};
	s_object* ps_tol = NEW_NUMERIC(1);
	NUMERIC_POINTER(ps_tol)[0] = dTol;
	s_object* ps_qrls = s_dqrls(GetPtr(), ps_y, ps_tol);
	if ( !SPL_NotThere(ps_qrls) )
	{
		for ( long i=0; i<__min(GET_LENGTH(ps_qrls),8); i++ )
			LIST_POINTER(ps_qrls)[i]->name = (char*)make_name(szNames[i], S_evaluator)->text;
	}

	return CSPmatrix(ps_qrls, TRUE);
}

//////////////////////// Matrix operations /////////////////////////////

////////////////////////////////////////////////////////////////////

// Matrix crossproduct
// *sC = sA' sB

//cross product: A'* B. if ps_B is NULL, return cross product of this matrix.
CSPmatrix CSPmatrix::crossprod(s_object* ps_B)
{
	s_object* ps_return= NULL;
	try
	{
		//Open top-level-eval if needed.
		CSPevaluator sEvaluator;
		//Convert data to "double before computating
		CSPmatrix sdA(GetPtr());
		CSPmatrix sdB;
		if(ps_B)
			sdB.Attach(ps_B);
		else
			sdB.Attach(GetPtr());

		if(sdA.GetDataMode() != S_MODE_DOUBLE)
			sdA.SetDataMode(S_MODE_DOUBLE);
		if(sdB.GetDataMode() != S_MODE_DOUBLE)
			sdB.SetDataMode(S_MODE_DOUBLE);

		long pnDA[2];
		pnDA[0] = sdA.GetNRow(FALSE);
		pnDA[1] = sdA.GetNCol(FALSE);

		long pnDB[2];
		pnDB[0] = sdB.GetNRow(FALSE);
		pnDB[1] = sdB.GetNCol(FALSE);

		if(pnDA[0] != pnDB[0])
			SCONNECT_ThrowException("Dimensions are incompatible");

		//Allocate the matrix to be returned.
		CSPmatrix sdC(pnDA[1], pnDB[1]);

		double* pA = (double*) sdA.GetElementPtr(0L, 0L, FALSE);
		double* pB = (double*) sdB.GetElementPtr(0L, 0L, FALSE);
		double* pC = (double*) sdC.GetElementPtr(0L, 0L, FALSE);

		//Matrix mult using Blas 3: sX = sX * sY
		F77_CALL(dcrossp2)(pA, pnDA, pB, pnDB, pC );

		//Make sure the returned object is valid
		ps_return = sEvaluator.CloneIfNeeded(sdC.Detach());
	}
	catch(CSPexception& except)
	{
		except.Print();
		ps_return = NULL;
	}
	catch(...)
	{
		ps_return = NULL;
	}
	return CSPmatrix(ps_return, TRUE);
}

////////////////////////////////////////////////////////////////////

// Matrix-matrix multiplication
// sC = sA sB

CSPobject SPL_Multiply(const CSPmatrix& sA, const CSPmatrix& sB)
{
	s_object* ps_return= NULL;
	try
	{
		if(!sA.IsValid())
			SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
		if(!sB.IsValid())
			SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

		//Open top-level-eval if needed.
		CSPevaluator sEvaluator;

		//Convert data to "double before computating
		CSPmatrix sdA(sA);
		CSPmatrix sdB(sB);
		if(sdA.GetDataMode() != S_MODE_DOUBLE)
			sdA.SetDataMode(S_MODE_DOUBLE);
		if(sdB.GetDataMode() != S_MODE_DOUBLE)
			sdB.SetDataMode(S_MODE_DOUBLE);

		long pnDA[2];
		pnDA[0] = sdA.GetNRow(FALSE);
		pnDA[1] = sdA.GetNCol(FALSE);

		long pnDB[2];
		pnDB[0] = sdB.GetNRow(FALSE);
		pnDB[1] = sdB.GetNCol(FALSE);

		if(pnDA[1] != pnDB[0])
			SCONNECT_ThrowException("Dimensions are incompatible");
		//Allocate the matrix to be returned.
		CSPmatrix sdC(pnDA[0], pnDB[1]);

		double* pA = (double*) sdA.GetElementPtr(0L, 0L, FALSE);
		double* pB = (double*) sdB.GetElementPtr(0L, 0L, FALSE);
		double* pC = (double*) sdC.GetElementPtr(0L, 0L, FALSE);

		//Matrix mult using Blas 3: sX = sX * sY
		F77_CALL(dmatmult)(pA, pnDA, pB, pnDB, pC );

		//Make sure the returned object is valid
		ps_return = sEvaluator.CloneIfNeeded(sdC.Detach());

	}
	catch(CSPexception& except)
	{
		except.Print();
		ps_return = NULL;
	}
	catch(...)
	{
		ps_return = NULL;
	}
	return CSPobject(ps_return, TRUE);
}

////////////////////////////////////////////////////////////////////

// Matrix-vector multiplication
// *sy = sA sx
// Return:
//		a numeric vector

CSPobject SPL_Multiply(const CSPmatrix& sA, const CSPnumeric& sx)
{
	s_object* ps_return= NULL;
	try
	{
		if(!sA.IsValid())
			SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);
		if(!sx.IsValid())
			SCONNECT_ThrowException(SCONNECT_INVALID_SOBJECT);

		//Convert data to "double before computating
		CSPmatrix sdA(sA);
		if(sdA.GetDataMode() != S_MODE_DOUBLE)
			sdA.SetDataMode(S_MODE_DOUBLE);

		long pnDA[2];
		pnDA[0] = sdA.GetNRow(FALSE);
		pnDA[1] = sdA.GetNCol(FALSE);

		long pnDX[2];
		pnDX[0] = sx.GetLength(FALSE);
		pnDX[1] = 1L;

		if(pnDX[0] == 1) //Scalar?
			ps_return = SPL_BinaryOp(sA, sx, S_BINOP_MULTIPLY);
		else if(pnDA[1] == pnDX[0])
		{
			//Create a new memory block for C matrix if neccessary
			ps_return = NEW_NUMERIC(pnDA[0]);

			double* pA = (double*) sdA.GetElementPtr(0L, 0L, FALSE);
			double* px = (double*) sx.GetRawPtr(FALSE);
			double* py = (double*) NUMERIC_POINTER(ps_return);

			//Matrix mult using Blas 3: sy = sA sx
			F77_CALL(dmatmult)(pA, pnDA, px, pnDX, py );
		}
		else 
			SCONNECT_ThrowException("Dimensions are incompatible");
	}
	catch(CSPexception& except)
	{
		except.Print();
		ps_return = NULL;
	}
	catch(...)
	{
		ps_return = NULL;
	}
	return CSPobject(ps_return, TRUE);

}

////////////////////////////////////////////////////////////////////

// Matrix backsolve
// Return:
//		a matrix

// matrix backsolve : R \ x. 
CSPobject CSPmatrix::backsolve(s_object* ps_x, long k)
{
	CSPmatrix sR(GetPtr());
	CSPmatrix sx(ps_x);
	s_object* ps_return= NULL;
	try
	{
		if(sR.GetDataMode() != S_MODE_DOUBLE || sx.GetDataMode()!=S_MODE_DOUBLE )
			SCONNECT_ThrowException("Data mode in back solve must be double");

		long nDR[2];
		nDR[0] = sR.GetNRow(FALSE);
		nDR[1] = sR.GetNCol(FALSE);
		if ( k <= 0 )
			k = nDR[1];

		if ( nDR[0] != nDR[1] )
			SCONNECT_ThrowException("R matrix must be square");

		long nDX[2];
		nDX[0] = sx.GetNRow(FALSE);
		nDX[1] = sx.GetNCol(FALSE);
		if( nDX[0] != nDR[1] )
			SCONNECT_ThrowException("Invalid number of rows %d in X", nDX[0]);

		double t = 0.;
		long i1 = 1;
	        long j1;
		for ( long j = k; j > 0; j-- )
		{
			if ( (double)sR(j,j) == 0.0 )
			{
				for ( long l=0; l<k; l++ )
					sx(j,l) = 0.0;
			}
			else
			{
				for ( long l=j; l<=k; l++ )
				{
					sx(j,l) = sx(j,l)/sR(j,j);
					if ( j > 1 )
					{
						t = -(double)sx(j,l);
						j1 = (long)(j-1);
						F77_CALL(daxpy)(&j1, &t, (double*)sR.GetElementPtr(0L, j-1, FALSE), &i1, 
										(double*)sx.GetElementPtr(0, l-1, FALSE), &i1);
					}
				}	
			}
		}
		ps_return = sx;
	}
	catch(CSPexception& except)
	{
		except.Print();
		ps_return = NULL;
	}
	catch(...)
	{ 
		ps_return = NULL;
	}
	return CSPobject(ps_return, TRUE);
}

//Apply the function to jth column vector 
BOOL CSPmatrix::IthRowApply(CSPmatrix& sAns /*in/out */, long i /*in:*/, const CSPcall& sfunCall /*in:*/)const
{
	BOOL bSuccess= TRUE;
	s_object* ps_vIthRow = sfunCall.GetArg(1L, FALSE);

	bSuccess = PutIthRowInto(ps_vIthRow, i); //put the jth col into the vector arg of FUN
	if(!bSuccess)
		return bSuccess;
	s_object* ps_vAnsIthRow = sfunCall.Eval(); //Apply to the new col.

	bSuccess = sAns.SetIthRowDirect(i, ps_vAnsIthRow);
	if(!bSuccess)
		return bSuccess;
	::try_to_free(ps_vAnsIthRow, S_TRUE, Nframe, S_evaluator);
	return bSuccess;
}

//Apply the function to jth column vector.
// pAns can change mode if new column required: logical->integer->single->double->complex->character
BOOL CSPmatrix::IthRowApply(CSPmatrix*& pAns /*in/out */, long i, const CSPcall& sfunCall)const
{
	BOOL bSuccess= TRUE;
	s_object* ps_vIthRow = sfunCall.GetArg(1L, FALSE);

	bSuccess = PutIthRowInto(ps_vIthRow, i); //put the jth col into the vector arg of FUN
	if(!bSuccess)
		return bSuccess;

	s_object* ps_value = sfunCall.Eval(); //Apply to the new row.

	//Success if NULL or VOID returned object and previous results were also the same
	if(SPL_NotThere(ps_value) && pAns->nrow(FALSE) == 0)
		return bSuccess;
	
	//Failure if previously good data, now NULL or VOID
	if(pAns->nrow(FALSE) > 0 && SPL_NotThere(ps_value))
		return FALSE;
	
	//Failure if pAns is not valid
	if((pAns == NULL) || !pAns->IsValid() || pAns->nrow(FALSE) == 0 || pAns->nrow(FALSE) <= i)
		return FALSE;

	long lLhsMode = pAns->GetDataMode();
	s_object* ps_vAnsIthRow = AS_VECTOR(ps_value);
	if(!IS_ATOMIC(ps_vAnsIthRow))
		return FALSE;

	long lReplacedMode = ::coerce_to(lLhsMode, ps_vAnsIthRow->mode, S_evaluator);
	if(lReplacedMode != lLhsMode)
	{
		char* pszMode = token_name(lReplacedMode,  S_evaluator);
		//Clone the whole object before modifying it.
		s_object* ps_newObject = pAns->Clone();
		::set_mode2(ps_newObject, lReplacedMode,  pszMode, S_evaluator);
		delete pAns;
		pAns = 	(CSPmatrix*)(SCONNECT_GenerateFromSObject(ps_newObject));
	}
	ps_vAnsIthRow = ::coeves(ps_vAnsIthRow, lReplacedMode, FALSE, CHECK_IT, &ps_vAnsIthRow, S_evaluator); 

	bSuccess = pAns->SetIthRowDirect(i, ps_vAnsIthRow);
	if(!bSuccess)
		return bSuccess;
	::try_to_free(ps_vAnsIthRow, S_TRUE, Nframe, S_evaluator);
	return TRUE;
}

//Apply the function to each column vector 
CSPobject CSPmatrix::RowApply(const CSPfunction& sfun, s_object* ps_list3Dots ) const
{
	BOOL bSuccess = TRUE;
	s_object* ps_ans=NULL;
	CSPmatrix* psAns=NULL;
	try
	{
		CSPevaluator sEvaluator;
		sEvaluator.PushFrame();
		{
			long nRow= GetNRow(FALSE);
			 
			//Build a function call object and setting all args including ... 
			CSPcall sfunCall(sfun.Clone()); 
			s_object *ps_arg1 = sEvaluator.alcvec(GetDataMode(FALSE), GetNCol(FALSE)) ;
			CSPcharacter sColNames = GetColNames(FALSE);
			if(sColNames.IsValid() && sColNames.GetLength()==ps_arg1->length) {
				s_object *ps_named = new_S_object(s_named_class, S_evaluator);
				ps_named->value.tree[0] = ps_arg1 ;
				ps_named->value.tree[1] = sEvaluator.CopyForWrite(sColNames.GetPtr()) ; /* does it need copying? Yes-otherwise SetPermRefCount fails to protect it */
				ps_arg1 = ps_named ;
			}
			sfunCall.SetArg(ps_arg1, 1L);
			if(!SPL_NotThere(ps_list3Dots) && GET_LENGTH(ps_list3Dots) > 0L)
			{
				const CSPlist slist3Dots(ps_list3Dots);
				if(slist3Dots.IsValid())
					sfunCall.AddArgs(CSPlist(slist3Dots.Clone(FALSE)));
			}
			//More efficient: make it perm. with respect to the current local eval frame
			sEvaluator.SetPermRefCount(&sfunCall, sEvaluator.GetCurrentEvalFramePtr());
			//Take care of the first column
			s_object* ps_vIthRow = sfunCall.GetArg(1L, FALSE); //Pointer to arg1 of the call object 
			bSuccess = PutIthRowInto(ps_vIthRow, 0L); //put the jth col into the vector arg of FUN
			if(!bSuccess)
				return CSPobject();
			s_object* ps_vFirstValue = sfunCall.Eval(); //Apply to the new col.
			long nAnsCol = 0;
			if(SPL_NotThere(ps_vFirstValue))
			{
				s_object* ps_object =  sEvaluator.eval("matrix(NULL)"); 
				psAns = new CSPmatrix(ps_object);
			}
			else
			{
				s_object* ps_vAnsIthRow = AS_VECTOR(ps_vFirstValue);
				if(!IS_ATOMIC(ps_vAnsIthRow))
					return CSPobject();
				nAnsCol = GET_LENGTH(ps_vAnsIthRow);
				if(nAnsCol < 1)
				{
				  s_object* ps_object =  sEvaluator.eval("matrix(NULL)"); 
				  psAns = new CSPmatrix(ps_object);
				}
				else
				{
				  CSPmatrix sAns(nRow, nAnsCol, ps_vAnsIthRow->mode); 
					if(SPL_NotThere(sAns.GetPtr()) || GET_LENGTH(sAns.GetPtr()) <= 0)
					  return CSPobject();
				  psAns = (CSPmatrix*)(SCONNECT_GenerateFromSObject(sAns.Detach()));
				  bSuccess = psAns->SetIthRowDirect(0L, ps_vAnsIthRow);
				  if(!bSuccess)
					  return CSPobject();
				}
			}

			//Main loop: push/pop new frames every kBlock
			const long kBlock=100L;
			long i=1L;
			for(long k=kBlock; k<=nRow; k+=kBlock)
			{
				sEvaluator.PushFrame();
				for(; i<k; ++i)
				{
					if(!IthRowApply(psAns, i, sfunCall))//adding result to *psAns
						return CSPobject();
				}
				sEvaluator.PopFrame();
				i=k;
			}			
			//Last set of loops: less then kBlock of loops
			sEvaluator.PushFrame();
			for(; i<nRow; ++i)
			{
				if(!IthRowApply(psAns, i, sfunCall))
					return CSPobject();
			}
			sEvaluator.PopFrame();
			s_object* ps_rowNames = NULL;
			CSPlist slistDimnames(GetDimnames(FALSE));
			if(slistDimnames.IsValid()) {
				long param = 0;
				ps_rowNames = slistDimnames.GetAt(param, FALSE);
			}
      if (psAns->GetNRow(FALSE)==0L)
      {
			//Sucsess with NULL return: return a matrix one element NULL to indecate a suscess.
        CSPmatrix sAns("matrix(NULL, 1, 1)");
				ps_ans = sAns.Detach();
			}
			else if (psAns->GetNCol(FALSE)==1L)
			{
				ps_ans = psAns->GetData(FALSE);
				if(!SPL_NotThere(ps_rowNames) && GET_LENGTH(ps_rowNames) > 0)
					SET_NAMES(ps_ans, ps_rowNames);
			}
			else
			{
				//Transpose to be consistent with the way S function apply()
				psAns->Transpose(FALSE);
				if(!SPL_NotThere(ps_rowNames) && GET_LENGTH(ps_rowNames) > 0)
					psAns->SetColNames(ps_rowNames, FALSE); //Because of transpose, prev ps_rowNames becomes colNames.
				if(IS_OBJ(ps_vFirstValue, named))
				{
					CSPnamed sNamed(ps_vFirstValue);
					if(sNamed.IsValid())
					{
						CSPcharacter scharFirstValueRowNames(sNamed.GetNames());
						if(scharFirstValueRowNames.IsValid() && scharFirstValueRowNames.length()==psAns->nrow(FALSE))
							psAns->SetRowNames(scharFirstValueRowNames, FALSE);
					}
				}
				else
				{
					CSPlist slistFirstValueDimNames(::get_dimnames(ps_vFirstValue));
					s_object* ps_rowNames= NULL;
					if(slistFirstValueDimNames.IsValid())
					{
						//Find the names from the first call to set to row names
						for(long iDimName=0; iDimName< slistFirstValueDimNames.length(); ++iDimName)
						{
							ps_rowNames = slistFirstValueDimNames.GetAt(iDimName);
							if(!SPL_NotThere(ps_rowNames) && GET_LENGTH(ps_rowNames) == psAns->GetNRow())
							{
								// This looks bogus, but I don't have an example of where it is used.  WWD.  Dec 2002.
								psAns->SetRowNames(ps_rowNames, FALSE);
								break;
							}
						}
					}
				}
				ps_ans = psAns->GetPtr();
			}
			psAns->Detach();
			delete psAns;
		}
		ps_ans = sEvaluator.PopFrame(ps_ans);
	}
	catch(...)
	{
		if(psAns != NULL)
		{
			psAns->Detach();
			delete psAns;
		}
		ps_ans = blt_in_NULL;
	}
	return CSPobject(ps_ans, TRUE);
}

//Apply the function to each column vector 
CSPobject CSPmatrix::RowApply(const char* pszCFun /* C function name */,
															s_object* ps_list3Dots) const
{
	BOOL bSuccess = TRUE;
	s_object* ps_ans=blt_in_NULL;
	CSPmatrix* psAns=NULL;
	if(!pszCFun || !*pszCFun)
		SCONNECT_ThrowException("Invalid input to CSPmatrix::RowApply()");
	try
	{
		CSPevaluator sEvaluator;
		sEvaluator.PushFrame();
		{
			//Get the C function pointer
			s_vfun_ptr pCFunc= (s_vfun_ptr)(sEvaluator.GetEntry(pszCFun));
			if(!pCFunc)
				SCONNECT_ThrowException("Cannot find S or C function: %s", pszCFun);
			//Setup args including list(...)
			long lArgs = 1L;
			if(!SPL_NotThere(ps_list3Dots) && GET_LENGTH(ps_list3Dots) > 0L)
			{
				const CSPlist slist3Dots(ps_list3Dots);
				if(slist3Dots.IsValid())
					lArgs += slist3Dots.GetLength(FALSE);
			}
			CSPlist slistArgs(lArgs);
			slistArgs.SetAt(0, ::alcvec(GetDataMode(FALSE), GetNCol(FALSE), S_evaluator));
			for(long n=1L; n< lArgs; n++)
				slistArgs.SetAt(n, LIST_POINTER(ps_list3Dots)[n-1L]);
			s_object* ps_vIthRow = slistArgs.GetAt(0); 

			//Take care of the first column
			bSuccess = PutIthRowInto(ps_vIthRow, 0L); //put the jth col into the vector arg of FUN
			if(!bSuccess)
				return CSPobject();
			s_object* ps_vAnsIthRow = ::do_call(pCFunc, const_cast<char*>(pszCFun), &ps_vIthRow, lArgs, S_evaluator);
			if(!ps_vAnsIthRow || GET_LENGTH(ps_vAnsIthRow) <1L)
				return CSPobject();
			//Creating the returned matrix object: the caller function must delete it!
			ps_vAnsIthRow = AS_VECTOR(ps_vAnsIthRow);
			if(!IS_ATOMIC(ps_vAnsIthRow))
				return CSPobject();
			long nRow= GetNRow(FALSE);
			CSPmatrix sAns(nRow, GET_LENGTH(ps_vAnsIthRow), ps_vAnsIthRow->mode); 
			if(SPL_NotThere(sAns.GetPtr()) || GET_LENGTH(sAns.GetPtr()) <= 0)
				return CSPobject();
			psAns = (CSPmatrix*)(SCONNECT_GenerateFromSObject(sAns.Detach()));

			bSuccess = psAns->SetIthRowDirect(0L, ps_vAnsIthRow);
			if(!bSuccess)
				return CSPobject();
			sEvaluator.TryToFree(ps_vAnsIthRow, S_TRUE);
			//Main loops:
			for(long i=1L; i< nRow; i++)
			{
				bSuccess = PutIthRowInto(ps_vIthRow, i); //put the jth col into the vector arg of FUN
				if(!bSuccess)
					return CSPobject();
				ps_vAnsIthRow = ::do_call(pCFunc, const_cast<char*>(pszCFun), &ps_vIthRow, lArgs, S_evaluator);
				bSuccess = psAns->SetIthRowDirect(i, ps_vAnsIthRow);
				if(!bSuccess)
					return CSPobject();
				sEvaluator.TryToFree(ps_vAnsIthRow, S_TRUE);
			}
			sEvaluator.TryToFree(ps_vIthRow);
			s_object* ps_rowNames = NULL;
			CSPlist slistDimnames(GetDimnames(FALSE));
			if(slistDimnames.IsValid()){
				long param = 0;
				ps_rowNames = slistDimnames.GetAt(param, FALSE);
			}

			if (psAns->GetNCol(FALSE)==1L)
			{
				ps_ans = psAns->GetData(FALSE);
				if(!SPL_NotThere(ps_rowNames) && GET_LENGTH(ps_rowNames) > 0)
					SET_NAMES(ps_ans, ps_rowNames);
			}
			else
			{
				//Transpose to be consistent with the way S function apply()
				psAns->Transpose(FALSE);

				if(!SPL_NotThere(ps_rowNames) && GET_LENGTH(ps_rowNames) > 0)
					psAns->SetRowNames(ps_rowNames, FALSE);
				ps_ans = psAns->GetPtr();
			}		
			psAns->Detach();
			delete psAns;
		}
		ps_ans = sEvaluator.PopFrame(ps_ans);
	}
	catch(...)
	{
		if(psAns != NULL)
		{
			psAns->Detach();
			delete psAns;
		}
		ps_ans = blt_in_NULL;
	}
	return CSPobject(ps_ans, TRUE);
}

//Apply the function to jth column vector 
BOOL CSPmatrix::JthColumnApply(CSPmatrix& sAns /*in/out */, long j, const CSPcall& sfunCall)const
{
	BOOL bSuccess = TRUE;
	s_object* ps_vJthColumn = sfunCall.GetArg(1L, FALSE);
	bSuccess = PutJthColumnInto(ps_vJthColumn, j); //put the jth col into the vector arg of FUN
	if(!bSuccess)
		return bSuccess;
	bSuccess = sAns.SetJthColumnDirect(j, ps_vJthColumn);
	if(!bSuccess)
		return bSuccess;
	::try_to_free(ps_vJthColumn, S_TRUE, Nframe, S_evaluator);
	return TRUE;
}

//Apply the function to jth column vector.
// pAns can change mode if new column required: logical->integer->single->double->complex->character
BOOL CSPmatrix::JthColumnApply(CSPmatrix*& pAns /*in/out */, long j, const CSPcall& sfunCall)const
{
	BOOL bSuccess = TRUE;
	s_object* ps_vJthColumn = sfunCall.GetArg(1L, FALSE);
	bSuccess = PutJthColumnInto(ps_vJthColumn, j); //put the jth col into the vector arg of FUN
	if(!bSuccess)
		return bSuccess;

	s_object* ps_value = sfunCall.Eval();
	//Success if NULL or VOID returned object and previous results were also the same
	if(SPL_NotThere(ps_value) && pAns->nrow(FALSE) == 0)
		return bSuccess;
	
	//Failure if previously good data, now NULL or VOID
	if(pAns->nrow(FALSE) > 0 && SPL_NotThere(ps_value))
		return FALSE;
	
	//Failure if pAns is not valid
	if((pAns == NULL) || !pAns->IsValid() || pAns->nrow(FALSE) == 0 || pAns->ncol(FALSE) <= j)
		return FALSE;

	s_object* ps_vAnsJthColumn = AS_VECTOR(ps_value); 
	if(!IS_ATOMIC(ps_vAnsJthColumn))
		return FALSE;
	long lLhsMode = pAns->GetDataMode();
	long lReplacedMode = ::coerce_to(lLhsMode, ps_vAnsJthColumn->mode, S_evaluator);
	if(lReplacedMode != lLhsMode)
	{
		char* pszMode = token_name(lReplacedMode,  S_evaluator);
		//Clone the whole object before modifying it.
		s_object* ps_newObject = pAns->Clone();
		::set_mode2(ps_newObject, lReplacedMode,  pszMode, S_evaluator);
		delete pAns;
		pAns = 	(CSPmatrix*)(SCONNECT_GenerateFromSObject(ps_newObject));
	}
	ps_vAnsJthColumn = ::coeves(ps_vAnsJthColumn, lReplacedMode, FALSE, CHECK_IT, &ps_vAnsJthColumn, S_evaluator); 

	bSuccess = pAns->SetJthColumnDirect(j, ps_vAnsJthColumn);
	if(!bSuccess)
		return bSuccess;
	::try_to_free(ps_vAnsJthColumn, S_TRUE, Nframe, S_evaluator);
	return TRUE;
}

//Apply the function to each column vector 
CSPobject CSPmatrix::ColumnApply(const CSPfunction& sfun, s_object* ps_list3Dots ) const
{
	BOOL bSuccess = TRUE;
	s_object* ps_ans=blt_in_NULL;
	CSPmatrix* psAns=NULL;
	try
	{
		CSPevaluator sEvaluator;
		sEvaluator.PushFrame();
		{
			long nCol= GetNCol(FALSE);
			s_object* ps_vJthColumn = ::alcvec(GetDataMode(FALSE), GetNRow(FALSE), S_evaluator); 
			CSPcharacter sRowNames = GetRowNames(FALSE);
			if(sRowNames.IsValid() && sRowNames.GetLength()>0) {
				s_object *ps_named = new_S_object(s_named_class, S_evaluator);
				ps_named->value.tree[0] = ps_vJthColumn ;
				ps_named->value.tree[1] = sEvaluator.CopyForWrite(sRowNames.GetPtr()) ; /* does it need copying? Yes-otherwise SetPermRefCount fails to protect it */
				ps_vJthColumn = ps_named ;
			}
			//Build a function call object and setting all args including ... 
			CSPcall sfunCall(sfun.Clone()); 
			sfunCall.SetArg(ps_vJthColumn, 1L);
			if(!SPL_NotThere(ps_list3Dots) && GET_LENGTH(ps_list3Dots) > 0L)
			{
				const CSPlist slist3Dots(ps_list3Dots);
				if(slist3Dots.IsValid())
					sfunCall.AddArgs(CSPlist(slist3Dots.Clone()));
			}

			//More efficient: make it perm. with respect to the current local eval frame
			sEvaluator.SetPermRefCount(&sfunCall, sEvaluator.GetCurrentEvalFramePtr());

			//Take care of the first column
			ps_vJthColumn = sfunCall.GetArg(1L, FALSE); //Pointer to arg1 of the call object 
			bSuccess = PutJthColumnInto(ps_vJthColumn, 0L); //put the jth col into the vector arg of FUN
			if(!bSuccess)
				return CSPobject();

			s_object* ps_vFirstValue = sfunCall.Eval(); //Apply to the new col.
			long nAnsRow = 0;
			if(SPL_NotThere(ps_vFirstValue))
			{
				s_object* ps_object =  sEvaluator.eval("matrix(NULL)"); 
				psAns = new CSPmatrix(ps_object);
			}
			else
			{
				s_object* ps_vAns1stColumn = AS_VECTOR(ps_vFirstValue);
				if(!IS_ATOMIC(ps_vAns1stColumn))
					return CSPobject();

				nAnsRow = GET_LENGTH(ps_vAns1stColumn);
				if(nAnsRow < 1)
				{
				  s_object* ps_object =  sEvaluator.eval("matrix(NULL)"); 
				  psAns = new CSPmatrix(ps_object);
				}
				else
				{
				  CSPmatrix sAns(nAnsRow, nCol, ps_vAns1stColumn->mode);
					if(SPL_NotThere(sAns.GetPtr()) || GET_LENGTH(sAns.GetPtr()) <= 0)
					  return CSPobject();
				  psAns = (CSPmatrix*)(SCONNECT_GenerateFromSObject(sAns.Detach()));
				  bSuccess = psAns->SetJthColumnDirect(0L, ps_vAns1stColumn);
				  if(!bSuccess)
					  return CSPobject();
				}
			}

			//Main loop: push/pop new frames every kBlock
			const long kBlock=100L;
			long j=1L;
			for(long k=kBlock; k<=nCol; k+=kBlock)
			{
				sEvaluator.PushFrame();
				for(; j<k; ++j) 
				{
					bSuccess = JthColumnApply(psAns, j, sfunCall); //adding result to *psAns
					if(!bSuccess)
						return CSPobject();
				}
				sEvaluator.PopFrame();
				j=k;
			}			
			
			sEvaluator.PushFrame();
			for(; j<nCol; ++j)
			{
				bSuccess = JthColumnApply(psAns, j, sfunCall);
				if(!bSuccess)
					return CSPobject();
			}
			sEvaluator.PopFrame();

			//Setting Dimnames if needed
			CSPlist slistDimnames(GetDimnames(FALSE));

			s_object* ps_colNames = NULL;
			if(slistDimnames.IsValid())
				ps_colNames = slistDimnames.GetAt(1L, FALSE);
      if (psAns->GetNRow(FALSE)==0L)
      {
			//Sucsess with NULL return: return a matrix one element NULL to indecate a suscess.
        CSPmatrix sAns("matrix(NULL, 1, 1)");
				ps_ans = sAns.Detach();
			}
			else if (psAns->GetNRow(FALSE)==1L)
			{
				ps_ans = psAns->GetData(FALSE);
				if(!SPL_NotThere(ps_colNames) && GET_LENGTH(ps_colNames) > 0)
					SET_NAMES(ps_ans, ps_colNames);
			}
			else
			{
				if(!SPL_NotThere(ps_colNames) && GET_LENGTH(ps_colNames) > 0)
					psAns->SetColNames(ps_colNames, FALSE);

				if(IS_OBJ(ps_vFirstValue, named))
				{
					CSPnamed sNamed(ps_vFirstValue);
					if(sNamed.IsValid())
					{
						CSPcharacter scharFirstValueRowNames(sNamed.GetNames());
						if(scharFirstValueRowNames.IsValid() && scharFirstValueRowNames.length()==psAns->nrow(FALSE))
							psAns->SetRowNames(scharFirstValueRowNames, FALSE);
					}
				}
				else
				{
					CSPlist slistFirstValueDimNames(::get_dimnames(ps_vFirstValue));
					s_object* ps_rowNames= NULL;
					if(slistFirstValueDimNames.IsValid())
					{
						//Find the names from the first call to set to row names
						for(long iDimName=0; iDimName< slistFirstValueDimNames.length(); ++iDimName)
						{
							ps_rowNames = slistFirstValueDimNames.GetAt(iDimName);
							if(!SPL_NotThere(ps_rowNames) && GET_LENGTH(ps_rowNames) == nAnsRow)
							{
								psAns->SetRowNames(ps_rowNames, FALSE);
								break;
							}
						}
					}
				}
				ps_ans = psAns->GetPtr();
			}
			psAns->Detach();
			delete psAns;		
		}
		ps_ans = sEvaluator.PopFrame(ps_ans);
	}
	catch(...)
	{
		if(psAns != NULL)
		{
			psAns->Detach();
			delete psAns;
		}
		ps_ans = blt_in_NULL;
	}
	return CSPobject(ps_ans, TRUE);
}

//Apply the function to each column vector 
CSPobject CSPmatrix::ColumnApply(const char* pszCFun /* C function name */, 
																 s_object* ps_list3Dots) const
{
	BOOL bSuccess=TRUE;
	s_object* ps_ans=NULL;
	CSPmatrix* psAns=NULL;
	if(!pszCFun || !*pszCFun)
		SCONNECT_ThrowException("Invalid input to CSPmatrix::ColumnApply()");
	try
	{
		CSPevaluator sEvaluator;
		sEvaluator.PushFrame();
		{
			//Get the C function pointer
			s_vfun_ptr pCFunc= (s_vfun_ptr)(sEvaluator.GetEntry(pszCFun));
			if(!pCFunc)
				SCONNECT_ThrowException("Cannot find S or C function: %s", pszCFun);
			//Setup args including list(...)
			long lArgs = 1L;
			if(!SPL_NotThere(ps_list3Dots) && GET_LENGTH(ps_list3Dots) > 0L)
			{
				const CSPlist slist3Dots(ps_list3Dots);
				if(slist3Dots.IsValid())
					lArgs += slist3Dots.GetLength(FALSE);
			}
			CSPlist slistArgs(lArgs);
			slistArgs.SetAt(0, sEvaluator.alcvec(GetDataMode(FALSE), GetNRow(FALSE)));
			for(long n=1L; n< lArgs; n++)
				slistArgs.SetAt(n, LIST_POINTER(ps_list3Dots)[n-1]);
			s_object* ps_vJthColumn = slistArgs.GetAt(0); 
			//Take care of the first column
			bSuccess = PutJthColumnInto(ps_vJthColumn, 0L); //put the jth col into the vector arg of FUN
			if(!bSuccess)
				return CSPobject();
			s_object*	ps_vAnsJthColumn = ::do_call(pCFunc, const_cast<char*>(pszCFun), &ps_vJthColumn, lArgs, S_evaluator);
			if(!ps_vAnsJthColumn || GET_LENGTH(ps_vAnsJthColumn) <1L)
				return CSPobject();
			//Creating the returned matrix object: the caller function must delete it!
			ps_vAnsJthColumn = AS_VECTOR(ps_vAnsJthColumn);
			if(!IS_ATOMIC(ps_vAnsJthColumn))
				return CSPobject();

			long nCol= GetNCol(FALSE);
			CSPmatrix sAns(GET_LENGTH(ps_vAnsJthColumn), nCol, ps_vAnsJthColumn->mode); 
			if(SPL_NotThere(sAns.GetPtr()) || GET_LENGTH(sAns.GetPtr()) <= 0)
				return CSPobject();

			psAns = (CSPmatrix*)(SCONNECT_GenerateFromSObject(sAns.Detach()));
			bSuccess = psAns->SetJthColumnDirect(0L, ps_vAnsJthColumn);
			if(!bSuccess)
				return CSPobject();
			sEvaluator.TryToFree(ps_vAnsJthColumn, S_TRUE);
			//Main loops:
			for(long j=1L; j< nCol; j++)
			{
				bSuccess = PutJthColumnInto(ps_vJthColumn, j); //put the jth col into the vector arg of FUN
				if(!bSuccess)
					return CSPobject();
				ps_vAnsJthColumn = ::do_call(pCFunc, const_cast<char*>(pszCFun), &ps_vJthColumn, lArgs, S_evaluator);

				bSuccess = psAns->SetJthColumnDirect(j, ps_vAnsJthColumn);
				if(!bSuccess)
					return CSPobject();
				sEvaluator.TryToFree(ps_vAnsJthColumn, S_TRUE);
			}
			sEvaluator.TryToFree(ps_vJthColumn);
			s_object* ps_colNames = NULL;
			CSPlist slistDimnames(GetDimnames(FALSE));
			if(slistDimnames.IsValid())
				ps_colNames = slistDimnames.GetAt(1L, FALSE);
			if (psAns->GetNRow(FALSE)==1L)
			{
				ps_ans = psAns->GetData(FALSE);
				if(!SPL_NotThere(ps_colNames) && GET_LENGTH(ps_colNames) > 0)
					SET_NAMES(ps_ans, ps_colNames);
			}
			else
			{
				if(!SPL_NotThere(ps_colNames) && GET_LENGTH(ps_colNames) > 0)
					psAns->SetColNames(ps_colNames, FALSE);
				ps_ans = psAns->GetPtr();
			}
			psAns->Detach();
			delete psAns;
		}
		ps_ans = sEvaluator.PopFrame(ps_ans);
	}
	catch(...)
	{
		if(psAns != NULL)
		{
			psAns->Detach();
			delete psAns;
		}
		ps_ans = blt_in_NULL;
	}
	return CSPobject(ps_ans, TRUE);
}

//Apply an S function to this matrix: FUN(v)
// lMargin = 1 : for each row i, call FUN(v) where v=A[i,] 
// lMargin = 2 : for each col j, call FUN(v) where v=A[,j] 
CSPobject CSPmatrix::Apply(long lMargin, s_object* ps_FUN, s_object* ps_list3Dots) const
{
	if(lMargin < 1L || 2L < lMargin)
		SCONNECT_ThrowException("lMargin out of range: %d", lMargin);
	// unused: s_object* ps_ans=NULL;
	CSPobject sAns;
	try
	{
		CSPevaluator sEvaluator;
		s_object* ps_fun_object = ps_FUN;
		switch(ps_FUN->mode)
		{
			case S_MODE_CHAR:
			{
				CSPcharacter scharFunName(ps_FUN);
				//Get the function to call
				s_object* ps_fun_object = sEvaluator.getFunction(scharFunName[0], TRUE, FALSE/* don't longjmp, it is ok if you can't find it*/);
				if(SPL_NotThere(ps_fun_object))
				{
					sAns = (lMargin == 1L )? RowApply(scharFunName[0]): ColumnApply(scharFunName[0]);
					break;
				}
			}
			default:
			{
				CSPfunction sFun(ps_fun_object); //validate the function object
				sAns = (lMargin == 1L )? RowApply(sFun, ps_list3Dots): ColumnApply(sFun, ps_list3Dots);
				break;
			}
		}
	}
	catch(...)
	{
		sAns.Attach(NULL);
	}
	return sAns;
}

//Apply an S function to this matrix: FUN(v, ps_list3Dots),
//where ps_listNamed3Dots are additional parameters passed via ...
CSPobject CSPmatrix::Apply(s_object* ps_Margin, s_object* ps_FUN, s_object* ps_list3Dots) const
{
	CSPevaluator sEvaluator;
	CSPinteger sintMargin(ps_Margin);
	return Apply(sintMargin(1), ps_FUN, ps_list3Dots);
}

///////////////////////////////////////////////////////////////////
////////////////// Matrix of character (double) /////////////////////

//Default constructor
CSPcharacterMatrix::CSPcharacterMatrix()
: TSPmatrix<char*, S_MODE_CHAR>()
{
}
//Copy constructor 
CSPcharacterMatrix::CSPcharacterMatrix(const CSPcharacterMatrix& sObject)
: TSPmatrix<char*, S_MODE_CHAR>()
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a base class object
CSPcharacterMatrix::CSPcharacterMatrix(const CSPobject& sObject)
: TSPmatrix<char*, S_MODE_CHAR>()
{	
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a valid S-expression
CSPcharacterMatrix::CSPcharacterMatrix(const char* pszExpression)
: TSPmatrix<char*, S_MODE_CHAR>(pszExpression)
{
}
//Construct from a valid S-object
CSPcharacterMatrix::CSPcharacterMatrix(s_object* ps_object, BOOL bTryToFreeOnDetach)
: TSPmatrix<char*, S_MODE_CHAR>()
{	
	Attach(ps_object, bTryToFreeOnDetach);
}
//Assigment from the same class
CSPcharacterMatrix& CSPcharacterMatrix::operator=(const CSPcharacterMatrix& sObject)
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
	return *this;
}
//Assigment from the base class
CSPcharacterMatrix& CSPcharacterMatrix::operator=(const CSPobject& sObject)
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
	return *this;
}
//Assigment from the S object
CSPcharacterMatrix& CSPcharacterMatrix::operator=(s_object* ps_object)
{
	Attach(ps_object, FALSE);
	return *this;
}
//The destructor
CSPcharacterMatrix::~CSPcharacterMatrix()
{
}
//////////////////// Other constructor/destructor and assignment operators
CSPcharacterMatrix::CSPcharacterMatrix(long nRow, long nCol)
: TSPmatrix<char*, S_MODE_CHAR>()
{
	Create(nRow, nCol, S_MODE_CHAR);
}

CSPcharacterMatrix::CSPcharacterMatrix(const CSPvector& sVector, long nRow, long nCol)
: TSPmatrix<char*, S_MODE_CHAR>()
{
	Create(sVector, nRow, nCol);
}

CSPcharacterMatrix::CSPcharacterMatrix(double* pdValues, long nRow, long nCol)
: TSPmatrix<char*, S_MODE_CHAR>()
{
	Create(pdValues, nRow, nCol);
}

//Copy content of one row to a vector
BOOL CSPcharacterMatrix::PutIthRowInto(char** pResult /*out: must pre-alloc */, long ithRow /*in: zero-based*/) const
{	
	return TSPmatrix<char*, S_MODE_CHAR>::PutIthRowInto(pResult, ithRow);
}

//Warning: ps_vResult shares strings with the .Data slot.  These strings will be destroyed or changed with this .Data.
BOOL CSPcharacterMatrix::PutIthRowInto(s_object* ps_vResult /*out: alloc if needed*/, long ithRow /*in: zero-based*/) const
{
	CSPevaluator sEvaluator;
	long nCol = GetNCol(FALSE);
	if(SPL_NotThere(ps_vResult))
		ps_vResult= sEvaluator.alcvec(S_MODE_CHAR, nCol); 
	else if( GET_LENGTH(ps_vResult)< nCol) 
		SET_LENGTH(ps_vResult, nCol);
	
	if(ps_vResult->mode != S_MODE_CHAR)
		ps_vResult = sEvaluator.coevec(ps_vResult, S_MODE_CHAR);

	TSPmatrix<char*, S_MODE_CHAR>::PutIthRowInto((char**)MyTypePointer(ps_vResult), ithRow);
	//Go with ATOMIC_WHY for quick sharing. 
	S_set_arena_why(ps_vResult, ATOMIC_WHY, S_evaluator);
	return TRUE;
}

//Copy content of one column to a vector
BOOL CSPcharacterMatrix::PutJthColumnInto( char** pResult /*out: must pre-alloc */, long jthCol /*in: zero-based*/) const
{		
	return TSPmatrix<char*, S_MODE_CHAR>::PutJthColumnInto(pResult, jthCol);
}
//Warning: ps_vResult shares strings with the .Data slot.  These strings will be destroyed or changed with this .Data.
BOOL CSPcharacterMatrix::PutJthColumnInto(s_object* ps_vResult /*out: alloc if needed*/, long jthCol /*in: zero-based*/) const
{
	CSPevaluator sEvaluator;
	long nRow = GetNRow(FALSE);
	if(SPL_NotThere(ps_vResult))
		ps_vResult= sEvaluator.alcvec(S_MODE_CHAR, nRow); 
	else if( GET_LENGTH(ps_vResult)< nRow) 
		SET_LENGTH(ps_vResult, nRow);		

	if(!IsMyMode(ps_vResult))
		ps_vResult = AsMyMode(ps_vResult);
	TSPmatrix<char*, S_MODE_CHAR>::PutJthColumnInto((char**)MyTypePointer(ps_vResult), jthCol);

	//Go with ATOMIC_WHY for quick sharing. 
	S_set_arena_why(ps_vResult, ATOMIC_WHY, S_evaluator);
	return TRUE;
}

//Set content of pValue into the .Data directly.
BOOL CSPcharacterMatrix::SetIthRowDirect(long ithRow /*in: zero-based*/, char** pValue /* in: must be valid*/)
{
	CSPevaluator sEvaluator;
	long nRow = GetNRow(FALSE);
	long nCol = GetNCol(FALSE);	
	if( ithRow<0L || nRow<=ithRow )
		return FALSE;
	//wrap CSPcharacter around the data 
	CSPcharacter schData(GetData(FALSE));
	long jCell=ithRow;
	for(long j=0L; j<nCol; ++j)
	{
		//Go with ATOMIC_WHY for quick sharing. 
		schData.SetAtDirect((char**) &pValue[j], jCell, jCell, FALSE, FALSE);
		jCell +=nRow;
	}
	DATA_SLOT(&(*this)) = schData.Detach(); //cautious: may not needed.

	return TRUE;
}

BOOL CSPcharacterMatrix::SetIthRowDirect(long ithRow /*in: zero-based*/, const s_object* ps_vector /* in: */)
{
	if(GetNCol(FALSE) != GET_LENGTH(ps_vector))
		return FALSE;
	//make sure string last at least as long as this object.
	CSPallocFrame sAllocFrame(GetStorageFrame(FALSE)); 
	CSPcharacter schRhs(SPL_Clone((s_object*) ps_vector, FALSE)); //make sure it is a character object
	return SetIthRowDirect(ithRow, ( char**) &schRhs[0]);
}

//Set content of ps_vector into the .Data directly .
BOOL CSPcharacterMatrix::SetJthColumnDirect(long jthCol /*in: zero-based*/,  char** pValue /* in: must be valid*/)
{		
	if( jthCol<0L || GetNCol(FALSE)<=jthCol )
		return FALSE; 

	CSPcharacter schData(GetData(FALSE));
	long nRow = GetNRow(FALSE);
	schData.SetAtDirect((char**) pValue, jthCol*nRow, (jthCol+1L)*nRow-1L, FALSE, FALSE);
	DATA_SLOT(&(*this)) = schData.Detach(); //cautious: may not needed.
	return TRUE;
}

BOOL CSPcharacterMatrix::SetJthColumnDirect(long jthCol /*in: zero-based*/, const s_object* ps_vector /* in: */)
{
	if (GetNRow(FALSE) != GET_LENGTH(ps_vector))
		return FALSE ;
	//make sure string last at least as long as this object.
	CSPallocFrame sAllocFrame(GetStorageFrame(FALSE)); 
	CSPcharacter schRhs(SPL_Clone((s_object*) ps_vector, FALSE)); //make sure it is a character object
	return SetJthColumnDirect(jthCol, ( char**) &schRhs[0]);
}

///////////////////////////////////////////////////////////////////
////////////////// Matrix of numeric (double) /////////////////////

//Default constructor
CSPnumericMatrix::CSPnumericMatrix()
: TSPmatrix<double, S_MODE_DOUBLE>()
{
}
//Copy constructor 
CSPnumericMatrix::CSPnumericMatrix(const CSPnumericMatrix& sObject)
: TSPmatrix<double, S_MODE_DOUBLE>()
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a base class object
CSPnumericMatrix::CSPnumericMatrix(const CSPobject& sObject)
: TSPmatrix<double, S_MODE_DOUBLE>()
{	
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
}
//Construct from a valid S-expression
CSPnumericMatrix::CSPnumericMatrix(const char* pszExpression)
: TSPmatrix<double, S_MODE_DOUBLE>(pszExpression)
{
}
//Construct from a valid S-object
CSPnumericMatrix::CSPnumericMatrix(s_object* ps_object, BOOL bTryToFreeOnDetach)
: TSPmatrix<double, S_MODE_DOUBLE>()
{	
	Attach(ps_object, bTryToFreeOnDetach);
}
//Assigment from the same class
CSPnumericMatrix& CSPnumericMatrix::operator=(const CSPnumericMatrix& sObject)
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
	return *this;
}
//Assigment from the base class
CSPnumericMatrix& CSPnumericMatrix::operator=(const CSPobject& sObject)
{
	Attach(&sObject, sObject.GetTryToFreeOnDetach());
	return *this;
}
//Assigment from the S object
CSPnumericMatrix& CSPnumericMatrix::operator=(s_object* ps_object)
{
	Attach(ps_object, FALSE);
	return *this;
}
//The destructor
CSPnumericMatrix::~CSPnumericMatrix()
{
}
//////////////////// Other constructor/destructor and assignment operators
CSPnumericMatrix::CSPnumericMatrix(long nRow, long nCol)
: TSPmatrix<double, S_MODE_DOUBLE>()
{
	Create(nRow, nCol, S_MODE_DOUBLE);
}

CSPnumericMatrix::CSPnumericMatrix(const CSPvector& sVector, long nRow, long nCol)
: TSPmatrix<double, S_MODE_DOUBLE>()
{
	Create(sVector, nRow, nCol);
}

CSPnumericMatrix::CSPnumericMatrix(double* pdValues, long nRow, long nCol)
: TSPmatrix<double, S_MODE_DOUBLE>()
{
	Create(pdValues, nRow, nCol);
}

//matrix-matrix multiply: %*% 
const CSPmatrix CSPnumericMatrix::Multiply(const CSPmatrix& sRhs) const
{
	return CSPmatrix::Multiply(sRhs);
}

//matrix-vector multiply: %*% : return 'matrix' if sRhs is scalar else  return 'vector' 
const CSPobject CSPnumericMatrix::Multiply(const CSPnumeric& sRhs) const
{
	return CSPmatrix::Multiply(sRhs);
}

//Call DGECON(LAPACK) to estimate the reciprocal of the condition number of a general
// real matrix A, in either the 1-norm or the infinity-norm, using
// the LU factorization computed by DGETRF.
// The condition number is computed as norm(A) * norm(inv(A)).

double CSPnumericMatrix::ConditionNumber(BOOL bValidate) const 
{
	return CSPmatrix::ConditionNumber(bValidate);
}

CSPmatrix CSPnumericMatrix::QRLS(s_object* ps_y, double dtol, BOOL bValidate) const
{
	return CSPmatrix::QRLS(ps_y, dtol, bValidate);
}

//cross product: A'* B. if ps_B is NULL, return cross product of this matrix.
CSPmatrix CSPnumericMatrix::crossprod(s_object* ps_B)
{
	return CSPmatrix::crossprod(ps_B);
}

// matrix backsolve : A \ B. If ps_B=NULL, returns inverse of A
CSPobject CSPnumericMatrix::backsolve(s_object* ps_B, long k)
{
	return CSPmatrix::backsolve(ps_B, k);
}

//returns sum of a numeric vector.
s_object* S_numeric_sum(s_object* ps_v)
{
	s_object* ps_value= blt_in_NULL;
	if(!ps_v)
		return ps_value;
	try
	{
		if(ps_v->mode == S_MODE_INT)
		{
			long lSum=0L;
			long* pVector = INTEGER_POINTER(ps_v);
			for(long i=0; i< GET_LENGTH(ps_v); ++i)
				lSum += pVector[i];
			ps_value=NEW_INTEGER(1);
			INTEGER_POINTER(ps_value)[0]= lSum;
		}
		else if(ps_v->mode == S_MODE_DOUBLE)
		{
			double dSum=0.0;
			double* pVector = NUMERIC_POINTER(ps_v);
			for(long i=0; i< GET_LENGTH(ps_v); ++i)
				dSum += pVector[i];
			ps_value=NEW_NUMERIC(1);
			NUMERIC_POINTER(ps_value)[0]= dSum;
		}
	}
	catch(...)
	{
		ps_value = blt_in_NULL;
	}
	return ps_value;
}

/*
S_matrix_apply() applies an S or C function to a matrix similar to lapply.
The S or C function is assumed to operate on a vector and return a new S object.
Prototype for the C function is 
	s_object myCFunction(s_object* ps_vector);

Usage:
mapply <- function(X, MARGIN, FUN, ...) 
{
	.Call("S_matrix_apply", X, MARGIN, FUN, list(...), COPY=c(F,F,F, F))
}
MARGIN=1
m<-100;n<-1000;X<-matrix(1.001:(m*n+.001),m,n)
print(sys.time({ans1<-apply(X, MARGIN, sum)}))
print(sys.time({ans2<-mapply(X, MARGIN, sum)}))
print(sys.time({ans3<-apply(X, MARGIN, function(v).Call("S_numeric_sum", v, COPY=F))}))
print(sys.time({ans4<-mapply(X, MARGIN, function(v).Call("S_numeric_sum", v, COPY=F))}))
print(sys.time({ans5<-mapply(X, MARGIN, "S_numeric_sum")}))
all.equal(ans1,ans2)
all.equal(ans1,ans3)
all.equal(ans1,ans4)
all.equal(ans1,ans5)

print(sys.time({ans1.2<-apply(X, MARGIN, sum, na.rm=T)}))
print(sys.time({ans2.2<-mapply(X, MARGIN, sum, na.rm=T)}))
all.equal(ans1.2,ans2.2)

f<-function(x) x
m<-100;n<-1000;X<-matrix(as.character(1:(m*n)),m,n)
.C("s_set_matrix_apply", FALSE) # old matrix apply
{mem.tally.reset();
 print(sys.time({ans10.1<-apply(X, MARGIN, f)}));
 print(mem.tally.report()[2])
}
.C("s_set_matrix_apply", TRUE) # new matrix apply
{mem.tally.reset();
 print(sys.time({ans10.2<-apply(X, MARGIN, f)}));
 print(mem.tally.report()[2])
}

*/
static long g_lS_matrix_apply = 1L ;
void s_set_matrix_apply(long *lFlag)
{
	long lTmp = *lFlag ;
	*lFlag = g_lS_matrix_apply ;
	if (lTmp>=0L)
		g_lS_matrix_apply = lTmp ;
}
s_object* S_matrix_apply(s_object* ps_X, s_object* ps_MARGIN, s_object* ps_FUN, s_object* ps_list3Dots)
{
	if((!g_lS_matrix_apply)
	|| (GET_LENGTH(ps_MARGIN) > 1L)
	|| (ps_FUN->mode != S_MODE_CHAR && ps_FUN->mode != S_MODE_FUN_DEF) )
		return blt_in_NULL; //silent return

	if(ps_X==NULL || !IS_OBJ(ps_X, matrix))
		return blt_in_NULL; //silent return



	s_object* ps_ans = blt_in_NULL;
	try
	{
		CSPevaluator sEvaluator; 	
		sEvaluator.PushFrame(); //push a new eval frame on the evaluator stack
		{//Ensure local C++ objects go out of scope before sEvaluator.PopFrame()
			CSPmatrix sX(ps_X); //validate
			switch(sX.GetDataMode())
			{
				case S_MODE_LGL:
				{
					CSPlogicalMatrix sMatrix(sX);
					ps_ans = sMatrix.Apply(ps_MARGIN, ps_FUN, ps_list3Dots);
					break;
				}
				case S_MODE_INT:
				{
					CSPintegerMatrix sMatrix(sX);
					ps_ans = sMatrix.Apply(ps_MARGIN, ps_FUN, ps_list3Dots);
					break;
				}
				case S_MODE_REAL:
				{
					CSPsingleMatrix sMatrix(sX);
					ps_ans = sMatrix.Apply(ps_MARGIN, ps_FUN, ps_list3Dots);
					break;
				}
				case S_MODE_DOUBLE:
				{
					CSPnumericMatrix sMatrix(sX);
					ps_ans = sMatrix.Apply(ps_MARGIN, ps_FUN, ps_list3Dots);
					break;
				}
				case S_MODE_CHAR:
				{
					CSPcharacterMatrix sMatrix(sX);
					ps_ans = sMatrix.Apply(ps_MARGIN, ps_FUN, ps_list3Dots);
					break;
				}
				case S_MODE_COMPLEX:
				{
					CSPcomplexMatrix sMatrix(sX);
					ps_ans = sMatrix.Apply(ps_MARGIN, ps_FUN, ps_list3Dots);
					break;
				}
				default:
				{
					ps_ans = blt_in_NULL;
					break;
				}
			} //switch
		}
		if(SPL_NotThere(ps_ans))
			ps_ans = blt_in_NULL;
		else
			ps_ans = sEvaluator.PopFrame(ps_ans); //pop and clear all objects in the local frame.
	}
	catch(...)
	{
		ps_ans = blt_in_NULL;
	}

	return ps_ans;
}

