/*
	Connect/C++ : Copyright (c) 2001, 2006 Insightful Corp.
	All rights reserved.
	Version 6.0: 2001
*/

#ifdef WIN32

#include <comdef.h>

#include "s.h"
#include "spvariant.h"

inline BOOL IsMissingValue(float value)
{
	return is_na_REAL( &value );
}

inline BOOL IsMissingValue(double value)
{
	return is_na_DOUBLE( &value );
}

inline BOOL IsMissingValue(long value)
{
	return is_na_INT( &value );
}

inline BOOL IsMissingValue(const char *cpszValue)
{
	return ( (!cpszValue || cpszValue[0] == '\0') ? TRUE : FALSE );
}

BOOL IsValidVariantType( const VARTYPE varType )
{
	BOOL bSuccess = TRUE;

	switch( varType )
	{
//		case VT_EMPTY:
		case VT_NULL:
			bSuccess = FALSE;
		break;
	}

	return bSuccess;
}

BOOL IsEmptyOrNullVariantType( const VARTYPE varType )
{
	BOOL bSuccess = TRUE;

	switch( varType )
	{
		case VT_EMPTY:
		case VT_NULL:
			bSuccess = FALSE;
		break;
	}

	return bSuccess;
}

LPVARIANT ConvertByRefVariantToVariant( const VARIANT &vaResult )
{
	LPVARIANT lpActualVariant = (LPVARIANT)(&vaResult);
	VARTYPE varType = lpActualVariant->vt;
	if ( varType & VT_BYREF )
	{
		switch ( varType & ~VT_BYREF )
		{
			case VT_VARIANT:
				lpActualVariant = vaResult.pvarVal;
			break;
		}
	}
	return lpActualVariant;
}

VARTYPE GetActualDataTypeFromVariant( const VARIANT &vaResult )
{
	VARTYPE varType = vaResult.vt;
	VARTYPE actualDataType = varType;

	if ( !IsValidVariantType( varType ) )
		return actualDataType;

	LPVARIANT lpActualVariant = ConvertByRefVariantToVariant( vaResult );
	if ( lpActualVariant )
		varType = lpActualVariant->vt;
	if ( !IsValidVariantType( varType ) )
		return actualDataType;

	if ( varType & VT_ARRAY )
	{
		VARTYPE arrayVarType = varType & ~VT_ARRAY;
		if ( !IsValidVariantType( arrayVarType ) )
			return actualDataType;
		varType = arrayVarType;

		if ( varType & VT_BYREF )
		{
			VARTYPE safeArrayVariantType = varType & ~VT_BYREF;
			if ( !IsValidVariantType( arrayVarType ) )
				return actualDataType;

			SAFEARRAY **ppSafeArray = lpActualVariant->pparray;
			if ( ppSafeArray )
			{
				SAFEARRAY *pSafeArray = *ppSafeArray;
				if ( pSafeArray )
				{
					if ( safeArrayVariantType == VT_VARIANT )
					{
						void *pvData = NULL;
						HRESULT hr = SafeArrayAccessData (pSafeArray, &pvData);
						if ( hr == S_OK )
						{
							VARIANT *pVariantElement = (VARIANT *)pvData;
							if ( pVariantElement )
							{
								// Try to get actual data type of elements in array by going through them
								// until a non-VT_EMPTY one is found and use it's data type (CRD)
								int numberOfElements = GetSafeArrayElementNumber( pSafeArray );
								for ( int i=0; i<numberOfElements; i++ )
								{
									safeArrayVariantType = pVariantElement[i].vt;

//									if ( safeArrayVariantType != VT_EMPTY )
									if ( safeArrayVariantType != VT_EMPTY && safeArrayVariantType != VT_ERROR )

										break;
								}	
							}

							if ( pSafeArray )
								SafeArrayUnaccessData (pSafeArray);
						}
					}
				}
			}
			actualDataType = safeArrayVariantType;
		}

		else if ( varType == VT_VARIANT )
		{
			VARTYPE safeArrayVariantType = VT_EMPTY;
			SAFEARRAY *pSafeArray = lpActualVariant->parray;
			if ( pSafeArray )
			{
				void *pvData = NULL;
				HRESULT hr = SafeArrayAccessData (pSafeArray, &pvData);
				if ( hr == S_OK )
				{
					VARIANT *pVariantElement = (VARIANT *)pvData;
					if ( pVariantElement )
					{
						// Try to get actual data type of elements in array by going through them
						// until a non-VT_EMPTY one is found and use it's data type (CRD)
						int numberOfElements = GetSafeArrayElementNumber( pSafeArray );
						for ( int i=0; i<numberOfElements; i++ )
						{
							safeArrayVariantType = pVariantElement[i].vt;

//							if ( safeArrayVariantType != VT_EMPTY )
							if ( safeArrayVariantType != VT_EMPTY && safeArrayVariantType != VT_ERROR )

								break;
						}
					}

					if ( pSafeArray )
						SafeArrayUnaccessData (pSafeArray);
				}
			}
			actualDataType = safeArrayVariantType;
		}

		// Else it's not a VARIANT so return the data type (CRD)
		else
			actualDataType = arrayVarType;

	}

	else if ( varType & VT_BYREF )
	{
		actualDataType = varType & ~VT_BYREF;
	}

	return actualDataType;
}	

// Routine to get total number of elements in a safearray (CRD)
int GetSafeArrayElementNumber( SAFEARRAY * pSafeArray )
{
	int numberOfElements = 1;
	int dimensions = SafeArrayGetDim( pSafeArray );
	for ( int i=0; i<dimensions; i++ )
	{
		long lBound = 0, uBound = 0;
		HRESULT hr = SafeArrayGetLBound(pSafeArray, i+1, &lBound);
		if ( hr != S_OK )
			break;
		hr = SafeArrayGetUBound(pSafeArray, i+1, &uBound);
		if ( hr != S_OK )
			break;
		long dimensionLength = uBound - lBound + 1;
		numberOfElements *= dimensionLength;
	}
	return numberOfElements;
}

SAFEARRAY * GetSafeArrayFromVariant( const LPVARIANT lpVariant )
{
	SAFEARRAY *pSafeArray = NULL;
	
	if ( !lpVariant )
		return pSafeArray;
		
	LPVARIANT lpActualVariant = ConvertByRefVariantToVariant( *lpVariant );
	if ( !lpActualVariant )
		return pSafeArray;

	VARTYPE varType = lpActualVariant->vt;
	if ( !IsValidVariantType( varType ) )
		return pSafeArray;
	
	if ( !(varType & VT_ARRAY) )
		return pSafeArray;
	
	if ( varType & VT_BYREF )
	{
		SAFEARRAY **ppSafeArray = lpActualVariant->pparray;
		if ( !ppSafeArray )
			return pSafeArray;
		pSafeArray = *ppSafeArray;
	}
	else
	{
		pSafeArray = lpActualVariant->parray;
	}
	return pSafeArray;
}	

int GetVariantDataSize( const VARTYPE varType )
{
	int nDataSize = 0;

	try
	{
		switch (varType)
		{
			case VT_I2:
				nDataSize = sizeof(short);
			break;

			case VT_I4:
				nDataSize = sizeof(long);
			break;

			case VT_R4:
				nDataSize = sizeof(float);
			break;

			case VT_R8:
			case VT_DATE:
				nDataSize = sizeof(double);
			break;

			case VT_CY:
				nDataSize = sizeof(CY);
			break;

			case VT_BSTR:
				nDataSize = sizeof( void * );
			break;

			case VT_DISPATCH:
				nDataSize = sizeof(DWORD);
			break;

			case VT_ERROR:
				nDataSize = sizeof(SCODE);
			break;

			case VT_BOOL:
				nDataSize = sizeof(BOOL);
			break;

			case VT_VARIANT:
				nDataSize = sizeof(VARIANT);
			break;

			case VT_UNKNOWN:
				nDataSize = sizeof(DWORD);
			break;

			case VT_ARRAY:
				nDataSize = sizeof( void * );
			break;
		
		}
	}
	catch(...)
	{
		nDataSize = 0;
	}

	return nDataSize;
}

BOOL ConvertVariantToVoidPtr( void **ppvRet, const VARIANT &vaResult )
{
	BOOL bSuccess = TRUE;

	if ( !ppvRet )
		return !bSuccess;

	try
	{
		// Initialize incoming return pointer values
		*ppvRet = NULL;

		// copy return value into return spot!
		switch (vaResult.vt)
		{
			case VT_I2:
				*ppvRet = S_ok_malloc( sizeof(long), S_evaluator );
				*(long*)(*ppvRet) = (long)(vaResult.iVal);
			break;

			case VT_I4:
				*ppvRet = S_ok_malloc( sizeof(long), S_evaluator );
				*(long*)(*ppvRet) = vaResult.lVal;
			break;

			case VT_R4:
				*ppvRet = S_ok_malloc( sizeof(float), S_evaluator );
				*(float *)(*ppvRet) = *(float *)&vaResult.fltVal;
			break;

			case VT_R8:
				*ppvRet = S_ok_malloc( sizeof(double), S_evaluator );
				*(double *)(*ppvRet) = *(double *)&vaResult.dblVal;
			break;

			case VT_DATE:
				*ppvRet = S_ok_malloc( sizeof(double), S_evaluator );
				*(double *)(*ppvRet) = *(double *)&vaResult.date;
			break;

			case VT_CY:
				*ppvRet = S_ok_malloc( sizeof(double), S_evaluator );
				if(VarR8FromCy(vaResult.cyVal, (double *)(*ppvRet)) != S_OK)
				{
					*(double *)(*ppvRet) = dNaN;
				}
			break;

			case VT_BSTR:
			{
				_bstr_t tmpBSTR(vaResult.bstrVal);
				int nResultLength = tmpBSTR.length();
				char *pszResult = (char *)S_ok_malloc( nResultLength+1, S_evaluator );
				strncpy( pszResult, (char *)tmpBSTR, nResultLength );
				pszResult[nResultLength] = 0;
				*ppvRet = (void *)pszResult;
			}
			break;

			// For dispatch returns, take the actual dispatch pointer and create one of our
			// OLE Automation client objects and return the handle to it (CRD)
			case VT_DISPATCH:
			{
				*ppvRet = S_ok_malloc( sizeof(DWORD), S_evaluator );
//				LPDISPATCH lpDispatch = vaResult.pdispVal;
//				*(DWORD*)(*ppvRet) = (DWORD)AutomationClient_CreateObjectFromDispatch( lpDispatch );
				*(DWORD*)(*ppvRet) = (DWORD)(vaResult.pdispVal);
			}
			break;

			case VT_ERROR:
				*ppvRet = S_ok_malloc( sizeof(SCODE), S_evaluator );
				*(SCODE*)(*ppvRet) = vaResult.scode;
			break;

			case VT_BOOL:
				*ppvRet = S_ok_malloc( sizeof(BOOL), S_evaluator );
				*(BOOL*)(*ppvRet) = (V_BOOL(&vaResult) != 0);
			break;

			case VT_VARIANT:
				*ppvRet = S_ok_malloc( sizeof(VARIANT), S_evaluator );
				*(VARIANT*)(*ppvRet) = vaResult;
			break;

			case VT_UNKNOWN:
				*ppvRet = S_ok_malloc( sizeof(DWORD), S_evaluator );
				*(LPUNKNOWN*)(*ppvRet) = vaResult.punkVal;
			break;

			default:
			{
				if ( (vaResult.vt & VT_ARRAY) == 0 )
				{
					bSuccess = FALSE;
					break;
				}

				SAFEARRAY *pSafeArray = vaResult.parray;
				if ( !pSafeArray )
					break;

				UINT nElementSize = SafeArrayGetElemsize(pSafeArray);
				if ( nElementSize <= 0 )
					break;

				UINT dimensions = SafeArrayGetDim( pSafeArray );
				if ( dimensions != 1 )
					break;

				long lBound1 = 0, uBound1 = 0;
				HRESULT hr = SafeArrayGetLBound(pSafeArray, 1, &lBound1);
				if ( hr != S_OK )
					break;
				hr = SafeArrayGetUBound(pSafeArray, 1, &uBound1);
				if ( hr != S_OK )
					break;

				long colLength = uBound1 - lBound1 + 1;
				if ( colLength <= 0 )
					break;

				void HUGEP *pvData = NULL;
				hr = SafeArrayAccessData (pSafeArray, &pvData);
				if ( hr != S_OK || !pvData )
					break;

				*ppvRet = S_ok_malloc( colLength, S_evaluator );
				memcpy( *ppvRet, pvData, colLength );

				// Release the ref on the safearray
				hr = SafeArrayUnaccessData (pSafeArray);
			}
			break;
		}
	}
	catch(...)
	{
		bSuccess = FALSE;
	}

	return bSuccess;
}

BOOL ConvertVoidPtrToVariant(
	LPVARIANT lpVariant,
	const void *vpData,
	const VARTYPE varType,
	const int cbDataSize,
	BOOL bMissingsAsErrors )
{
	BOOL bSuccess = TRUE;

	if ( !lpVariant )
		return !bSuccess;
	if ( !vpData )
		return !bSuccess;

	va_list argList = (va_list)vpData;

	lpVariant->vt = varType; // set the variant type
	if (lpVariant->vt & VT_MFCBYREF)
	{
		lpVariant->vt &= ~VT_MFCBYREF;
		lpVariant->vt |= VT_BYREF;
	}

	switch (lpVariant->vt)
	{
		case VT_I2:
			lpVariant->iVal = va_arg(argList, short);
		break;

		case VT_I4:
		{
			long value = va_arg(argList, long);
			if ( IsMissingValue(value) && bMissingsAsErrors )
			{
				lpVariant->vt = VT_ERROR;
				lpVariant->scode = 0x800A07FA;
			}
			else
			{
				lpVariant->lVal = value;
			}
		}

		break;

		case VT_R4:
			// Note: All float arguments to vararg functions are passed
			//  as doubles instead.  Thats why they are passed as VT_R8
			//  instead of VT_R4.
		{
			float value = va_arg(argList, float);
			if ( IsMissingValue(value) && bMissingsAsErrors )
			{
				lpVariant->vt = VT_ERROR;
				lpVariant->scode = 0x800A07FA;
			}
			else
			{
				lpVariant->vt = VT_R8;
				*(double*)&lpVariant->dblVal = value;
			}
		}

		break;

		case VT_R8:
		{
			double value = va_arg(argList, double);
			if ( IsMissingValue(value) && bMissingsAsErrors )
			{
				lpVariant->vt = VT_ERROR;
				lpVariant->scode = 0x800A07FA;
			}
			else
			{
				lpVariant->vt = VT_R8;
				*(double*)&lpVariant->dblVal = value;
			}
		}

		break;

		case VT_DATE:
		{
			double value = va_arg(argList, double);
			if ( IsMissingValue(value) && bMissingsAsErrors )
			{
				lpVariant->vt = VT_ERROR;
				lpVariant->scode = 0x800A07FA;
			}
			else
			{
				lpVariant->vt = VT_DATE;
				*(double*)&lpVariant->date = value;
			}
		}

		break;

		case VT_CY:
			lpVariant->cyVal = *va_arg(argList, CY*);
		break;

		case VT_BSTR:
		{
			BSTR bstr = va_arg(argList, BSTR);
			_bstr_t tmpBSTR(bstr);
			if ( IsMissingValue((char *)tmpBSTR) && bMissingsAsErrors )
			{
				lpVariant->vt = VT_ERROR;
				lpVariant->scode = 0x800A07FA;
			}
			else
			{
				lpVariant->bstrVal = ::SysAllocString(bstr);
				if (bstr != NULL && lpVariant->bstrVal == NULL)
					throw(NULL);
				lpVariant->vt = VT_BSTR;
			}
		}
		break;

		case VT_BSTRA:
		{
			LPCSTR lpsz = va_arg(argList, LPSTR);
			if ( IsMissingValue(lpsz) && bMissingsAsErrors )
			{
				lpVariant->vt = VT_ERROR;
				lpVariant->scode = 0x800A07FA;
			}
			else
			{
				_bstr_t tmpBSTR(lpsz);
				lpVariant->bstrVal = tmpBSTR.copy();
				if (lpsz != NULL && lpVariant->bstrVal == NULL)
					throw(NULL);
				lpVariant->vt = VT_BSTR;
			}
		}
		break;

		case VT_DISPATCH:
			lpVariant->pdispVal = va_arg(argList, LPDISPATCH);
		break;

		case VT_ERROR:
			lpVariant->scode = va_arg(argList, SCODE);
		break;

		case VT_BOOL:
			V_BOOL(lpVariant) = (VARIANT_BOOL)(va_arg(argList, BOOL) ? -1 : 0);
		break;

		case VT_VARIANT:
			*lpVariant = *va_arg(argList, VARIANT*);
		break;

		case VT_UNKNOWN:
			lpVariant->punkVal = va_arg(argList, LPUNKNOWN);
		break;

		case VT_I2|VT_BYREF:
			lpVariant->piVal = va_arg(argList, short*);
		break;

		case VT_I4|VT_BYREF:
			lpVariant->plVal = va_arg(argList, long*);
		break;

		case VT_R4|VT_BYREF:
			lpVariant->pfltVal = va_arg(argList, float*);
			break;

		case VT_R8|VT_BYREF:
			lpVariant->pdblVal = va_arg(argList, double*);
			break;

		case VT_DATE|VT_BYREF:
			lpVariant->pdate = va_arg(argList, DATE*);
			break;

		case VT_CY|VT_BYREF:
			lpVariant->pcyVal = va_arg(argList, CY*);
			break;

		case VT_BSTR|VT_BYREF:
			lpVariant->pbstrVal = va_arg(argList, BSTR*);
			break;

		case VT_DISPATCH|VT_BYREF:
			lpVariant->ppdispVal = va_arg(argList, LPDISPATCH*);
			break;

		case VT_ERROR|VT_BYREF:
			lpVariant->pscode = va_arg(argList, SCODE*);
			break;

		case VT_BOOL|VT_BYREF:
		{
			// coerce BOOL into VARIANT_BOOL
			BOOL* pbool = va_arg(argList, BOOL*);
			*pbool = *pbool ? MAKELONG(-1, 0) : 0;
			lpVariant->pboolVal = (VARIANT_BOOL*)pbool;
		}
		break;

		case VT_VARIANT|VT_BYREF:
			lpVariant->pvarVal = va_arg(argList, VARIANT*);
		break;

		case VT_UNKNOWN|VT_BYREF:
			lpVariant->ppunkVal = va_arg(argList, LPUNKNOWN*);
		break;

		case VT_ARRAY:
		{
			if ( cbDataSize <= 0 )
			{
				bSuccess = FALSE;
				break;
			}

			// Allocate and init safearray
			SAFEARRAYBOUND aDim[1]; 
			aDim[0].lLbound = 0; 
			aDim[0].cElements = cbDataSize;

			SAFEARRAY *pArray = SafeArrayCreate(VT_UI1, 1, aDim);
			if ( pArray )
			{
				// Lock down data in safe array and get it into a pointer
				void *pvData = NULL;
				HRESULT hr = SafeArrayAccessData (pArray, &pvData);
				if ( hr == S_OK && pvData )
				{
					void *pIn = va_arg(argList, void *);
					memcpy( pvData, pIn, cbDataSize );
					
					hr = SafeArrayUnaccessData (pArray);
					if ( hr == S_OK )
						lpVariant->parray = pArray;

					// Adjust the variant data type to an array of BYTEs (CRD)
					lpVariant->vt = VT_ARRAY|VT_UI1;
				}
			}
		}
		break;

		default:
			bSuccess = FALSE;
		break;
	}

	return bSuccess;
}

BOOL ConvertVariantArrayToByteArray(
	char **ppByteArray,
	const VARIANT *pVariantArray,
	const long variantArrayLength )
{
	BOOL bSuccess = FALSE;

	if ( !ppByteArray )
		return bSuccess;
	if ( !pVariantArray || variantArrayLength <= 0 )
		return bSuccess;

	// Initialize incoming return pointer values
	*ppByteArray = NULL;

	try
	{
		int nByteArraySize = 0;
		long i=0;

		for ( i=0; i<variantArrayLength; i++ )
		{
			int nSizeElement = GetVariantDataSize( pVariantArray[i].vt );
			nByteArraySize += nSizeElement;
		}

		*ppByteArray = new char [nByteArraySize];
		memset( *ppByteArray, 0, nByteArraySize );

		int nCurrentPosition = 0;
		for ( i=0 ; i<variantArrayLength; i++ )
		{
			void *pvVariantDataElement = NULL;
			if ( !ConvertVariantToVoidPtr( &pvVariantDataElement, pVariantArray[i] ) ||
				  !pvVariantDataElement )
				continue;

			int nSizeElement = GetVariantDataSize( pVariantArray[i].vt );
			if ( nSizeElement > 0 )
			{
				if ( (pVariantArray[i]).vt == VT_BSTR )
					memcpy( (*ppByteArray)+nCurrentPosition, &pvVariantDataElement, nSizeElement );

				else if ( (pVariantArray[i]).vt == VT_ARRAY )
					memcpy( (*ppByteArray)+nCurrentPosition, &pvVariantDataElement, nSizeElement );

				else
					memcpy( (*ppByteArray)+nCurrentPosition, pvVariantDataElement, nSizeElement );

				nCurrentPosition += nSizeElement;
			}

			if ( (pVariantArray[i]).vt != VT_BSTR && (pVariantArray[i].vt) != VT_ARRAY )
			{
				delete pvVariantDataElement;
				pvVariantDataElement = NULL;
			}
		}

		bSuccess = TRUE;
	}
	catch(...)
	{
		bSuccess = FALSE;
	}

	return bSuccess;
}

BOOL ConvertByteArrayToVariantArray(
	LPVARIANT pVariantArray,
	va_list pByteArray,
	const long variantArrayLength,
	int *pDataSizes,
	BOOL bMissingsAsErrors )
{
	BOOL bSuccess = FALSE;

	if ( !pByteArray )
		return bSuccess;
	if ( !pVariantArray || variantArrayLength <= 0 )
		return bSuccess;
	if ( !pDataSizes )
		return bSuccess;

	try
	{
		int nCurrentPosition = 0;
		for ( int i=0 ; i<variantArrayLength; i++ )
		{
			if ( !ConvertVoidPtrToVariant(
					&pVariantArray[i],
					pByteArray+nCurrentPosition,
					(pVariantArray[i]).vt,
					pDataSizes[i],
					bMissingsAsErrors ) )
				continue;

			int nSizeElement = pDataSizes[i];
			if ( nSizeElement > 0 )
				nCurrentPosition += nSizeElement;
		}

		bSuccess = TRUE;
	}
	catch(...)
	{
		bSuccess = FALSE;
	}

	return bSuccess;
}

HRESULT TxCopyVariant( LPVARIANT lpDestVariant, const LPVARIANT lpSourceVariant )
{
	HRESULT hr = S_FALSE;
	
	if ( !lpDestVariant || !lpSourceVariant )
		return hr;
	
	hr = VariantCopyInd( lpDestVariant, lpSourceVariant );	
	return hr;
}	

HRESULT TxDeleteVariant( VARIANT **ppVariant, BOOL bDeleteVariant )
{
	HRESULT hr = S_FALSE;
	if ( !ppVariant )
		return hr;
	
	if ( !*ppVariant )	
		return S_OK;

	// If the variant is VT_BYREF, then nuke the ref'd pointer if appropriate (CRD)
	LPVARIANT lpVariant = *ppVariant;
	if ( lpVariant->vt & VT_BYREF )
	{
		VARTYPE subType = lpVariant->vt & ~VT_BYREF;
		switch (subType)
		{
			case VT_I2:
				delete lpVariant->piVal;
				break;

			case VT_I4:
				delete lpVariant->plVal;
				break;

			case VT_R4:
				delete lpVariant->pfltVal;
				break;

			case VT_R8:
				delete lpVariant->pdblVal;
				break;

			case VT_DATE:
				delete lpVariant->pdate;
				break;

			case VT_CY:
				delete lpVariant->pcyVal;
				break;

			case VT_BSTR:
				delete lpVariant->pbstrVal;
				break;

			case VT_ERROR:
				delete lpVariant->pscode;
				break;

			case VT_BOOL:
			{
				delete lpVariant->pboolVal;
			}
			break;

			case VT_VARIANT:
				VariantClear( lpVariant->pvarVal );
				delete lpVariant->pvarVal;
			break;
		}
	}

	hr = VariantClear( *ppVariant );
	if ( bDeleteVariant )
	{
		delete *ppVariant;
		*ppVariant = NULL;
	}
	return hr;	
}	

LPVARIANT ConvertSafeArrayToVariantSafeArray( LPVARIANT lpInVariant )
{
	LPVARIANT lpReturnVariant = NULL;
	
	if ( !lpInVariant )
		return lpReturnVariant;

	VARTYPE inputVariantType = lpInVariant->vt;
	if ( !IsValidVariantType( inputVariantType ) )
		return lpReturnVariant;

	LPVARIANT lpActualVariant = ConvertByRefVariantToVariant( *lpInVariant );
	if ( !lpActualVariant )
		return lpReturnVariant;

	inputVariantType = lpActualVariant->vt;
	if ( !IsValidVariantType( inputVariantType ) )
		return lpReturnVariant;

	VARTYPE actualDataType = GetActualDataTypeFromVariant( *lpActualVariant );
	if ( !IsValidVariantType( actualDataType ) )
		return lpReturnVariant;

	if ( inputVariantType & VT_ARRAY )
	{
		VARTYPE arrayDataType = inputVariantType & ~VT_ARRAY;
		if ( !IsValidVariantType( arrayDataType ) )
			return lpReturnVariant;

		if ( arrayDataType & VT_BYREF )
			inputVariantType = arrayDataType & ~VT_BYREF;
		else	
			inputVariantType = arrayDataType;
	}
	else
		return lpReturnVariant;
	
	if ( inputVariantType == VT_VARIANT )	
		return lpReturnVariant;

	SAFEARRAY *pSafeArray = GetSafeArrayFromVariant( lpActualVariant );
	if ( !pSafeArray )
		return lpReturnVariant;

	UINT dimensions = SafeArrayGetDim( pSafeArray );
	if ( dimensions > 2 || dimensions < 1 )
		return lpReturnVariant;

	long lBound1 = 0, uBound1 = 0;
	long lBound2 = 0, uBound2 = 0;
	HRESULT hr = SafeArrayGetLBound(pSafeArray, 1, &lBound1);
	if ( hr != S_OK )
		return lpReturnVariant;
	hr = SafeArrayGetUBound(pSafeArray, 1, &uBound1);
	if ( hr != S_OK )
		return lpReturnVariant;
	if ( dimensions == 2 )
	{
		hr = SafeArrayGetLBound(pSafeArray, 2, &lBound2);
		if ( hr != S_OK )
			return lpReturnVariant;
		hr = SafeArrayGetUBound(pSafeArray, 2, &uBound2);
		if ( hr != S_OK )
			return lpReturnVariant;
	}

	long nColLength = uBound1 - lBound1 + 1;
	long nCols = uBound2 - lBound2 + 1;
	if ( nColLength <= 0 || nCols <= 0 )
		return lpReturnVariant;

	void HUGEP *pvData = NULL;
	hr = SafeArrayAccessData (pSafeArray, &pvData);
	if ( hr != S_OK || !pvData )
		return lpReturnVariant;

	int nElementSize = SafeArrayGetElemsize(pSafeArray);

	int *pOriginalDataSizes = NULL;
	SAFEARRAYBOUND *paDim = NULL;
	SAFEARRAY *psaNewSafeArray = NULL;
	LPVARIANT lpVariantArray = NULL;

	try
	{
		long i=0;

		lpVariantArray = new VARIANT[nCols * nColLength];
		memset( lpVariantArray, 0, (nCols * nColLength) * sizeof(VARIANT) );
		pOriginalDataSizes = new int[nCols * nColLength];
		for ( i=0; i<nCols*nColLength; i++ )
		{
			lpVariantArray[i].vt = inputVariantType;
			pOriginalDataSizes[i] = nElementSize;
		}
		if ( !ConvertByteArrayToVariantArray(
				lpVariantArray,
				(va_list)pvData,
				nCols * nColLength,
				pOriginalDataSizes ) )
			throw(NULL);

		paDim = new SAFEARRAYBOUND[dimensions]; 
		paDim[0].lLbound = 1; 
		paDim[0].cElements = nColLength;
		if ( dimensions >= 2 )
		{
			paDim[1].lLbound = 1; 
			paDim[1].cElements = nCols;
		}

		// Allocate and init safearray
		psaNewSafeArray = SafeArrayCreate(VT_VARIANT, dimensions, paDim);
		if ( !psaNewSafeArray )
			throw(NULL);

		// Lock down data in safe array and get it into a pointer
		void *pvData = NULL;
		hr = SafeArrayAccessData (psaNewSafeArray, &pvData);
		if ( hr != S_OK || !pvData )
			throw(NULL);

//		memcpy( pvData, lpVariantArray, (nCols * nColLength) * sizeof(VARIANT) );
		VARIANT *pTargetVariantArray = (VARIANT *)pvData;
		for ( i=0; i<nCols*nColLength; i++ )
		{
			TxCopyVariant( &(pTargetVariantArray[i]), &(lpVariantArray[i]) );
		}

		// Release the ref on the safearray
		hr = SafeArrayUnaccessData (psaNewSafeArray);
		if ( hr != S_OK )
			throw(NULL);

		// Create the return variant
		lpReturnVariant = new VARIANT;
		lpReturnVariant->vt = VT_ARRAY | VT_VARIANT;
		lpReturnVariant->parray = psaNewSafeArray;
	}
	catch(...)
	{
		if ( lpReturnVariant )
		{
			TxDeleteVariant( &lpReturnVariant, TRUE );
		}
	}

	if ( lpVariantArray )
	{
		for ( int i=0; i<nCols*nColLength; i++ )
		{
			LPVARIANT lpElement = &(lpVariantArray[i]);
			TxDeleteVariant( &lpElement, FALSE );
		}
		delete [] lpVariantArray;

		lpVariantArray = NULL;
	}
	if ( pOriginalDataSizes )
	{
		delete pOriginalDataSizes;
		pOriginalDataSizes = NULL;
	}
	if ( paDim )
	{
		delete [] paDim;
		paDim = NULL;
	}

	hr = SafeArrayUnaccessData (pSafeArray);

	return lpReturnVariant;
}	

BOOL IsVariant2D( const LPVARIANT lpVariant )
{
	BOOL bIs2D = FALSE;
	
	SAFEARRAY *pSafeArray = GetSafeArrayFromVariant( lpVariant );
	if ( !pSafeArray )
		return bIs2D;
	bIs2D = ( SafeArrayGetDim( pSafeArray ) == 2 ) ? TRUE:FALSE;
	return bIs2D;
}	

#endif // WIN32
