package Splus::DATAINSTALL;
use v5.8;
use strict ;

=head1 NAME

  Splus::DATAINSTALL - install S code files to a given package/.Data directory

=head1 SYNOPSIS

  The following is the plan.  Only the ones before the bar are implemented.

  use Splus::DATAINSTALL;
  $sh = Splus::DATAINSTALL->new(@ARGV) ; # args: DESTDIR file1.txt file2.csv ...
  $sh->make() ; # do the installation
  --- bar ---
  $sh->make_virgin() ; # remove all generated files, including destdir
  $sh->make_from_scratch(); # make_virgin then make
  $sh->add_source_files("foo.q"); # add source file to list
  $sh->delete_source_files("foo.q"); # remove source file from list
  $sh->get_source_files("foo.q"); # return list of *.txt *.csv files
  $sh->tidy_after_build() ; # do nothing.  Supplied to be like SHLIB.

=cut

use Splus::Vars;
use Splus::Utils;
use Splus::SplusUtils;
use Cwd qw(getcwd abs_path) ;

use Data::Dumper;

sub _Syntax {
    print "-h,--help             # print this help message\n";
    print "--destdir dirName     # install to dirName (don't include .Data)\n";
    print "--clean-first         # remove things to build, if they exist, before rebuild\n";
    print "obj1.txt obj2.csv     # files to read with read.table (csv=>sep=';',txt=>sep='')\n";
    @_ && die join(", ", @_) ;
}

sub new {
    # Converts data files (*.txt, etc.) in current directory
    # to S objects in DESTDIR/.Data
    # $tmp = Splus::DATAINSTALL::new qw(--destdir DESTDIR --clean-first file1.txt file2.csv)
    # sets things up and $tmp->make() does the installation.
    # If no files supplied then include all *.txt, *.csv, etc.
    my $class = shift ;
    my $self = {} ;
    my $ret = bless $self, $class ;
    $self->{srcs} = {};
    $self->{src_id} = 0 ;
    my $src_supplied = 0 ;

    while (@_) {
        $_ = shift @_ ;
        if (/^(-h|--help)/) {
            _Syntax() ;
            exit() ;
        } elsif (/^(-v|--verbose)/) {
            $self->{verbose} = 1 ;
        } elsif (/^--clean-first/) {
            $self->{clean_first} = 1 ;
        } elsif (/^--clean-after/) { # there is no clean_after action, but supply argument for consistency
            $self->{clean_after} = 1 ;
        } elsif (/^--no-clean-after/) {
            $self->{clean_after} = 0 ;
        } elsif (/^--destdir/) {
            $self->{destdir} = shift @_ or _Syntax("No directory name after --destdir");
            $self->_canonicalize_destdir() ;
        } else {
            $self->add_source_files($_) ;
            $src_supplied = 1 ;
        }
    }
    $self->{destdir} or _Syntax "No --destdir directory given" ;
    if (!$src_supplied) {
        # This globbing can be done in Splus code now
        $self->add_source_files(globlist(qw(*.txt *.TXT *.csv *.CSV *.q *.R *.ssc *.SSC *.S *.tab *.TAB *.Rdata *.rdata *.rda))) ;
    }
    $ret ;
}

sub _mkdir
{
    # not for public use.
    # make a directory if it does not exist.  Die if problems arise.
    my $dir = shift ;
    if (! -d $dir ) {
       die "A non-directory file $dir exists and we need to make a directory by that name" if -e $dir ;
       mkdir $dir or die "Cannot make directory $dir ($!)" ;
    }
}

sub _prepare_destdir
{
    # make sure destdir has a .Data with right components in it.
    # Don't touch a good one, but fix up incomplete ones.
    # Die if destdir itself does not exist or if there are
    # any problems making .Data and its subdirectories.
    # Should have __Hhelp/  __Meta/  __Shelp/
    # and empty file ___nonfile (unix) or ___nonfi (windows or windows-compatible).
    # For now, we will leave out the nonfile.
    my $self = shift ;
    my $destdir = $self->{destdir} ;
    _Syntax "Destination directory $destdir does not exist or is not a directory" if ! -d $destdir ;
    _mkdir ("$destdir/.Data") ;
    _mkdir ("$destdir/.Data/__Meta") ;
    _mkdir ("$destdir/.Data/__Hhelp") ;
    _mkdir ("$destdir/.Data/__Shelp") ;
    -w "$destdir/.Data" or die abs_path("$destdir/.Data") . " is not writable" ;
    -w "$destdir/.Data/__Meta" or die abs_path("$destdir/.Data/__Meta") . " is not writable" ;
}

sub _clean_destdir_data
{
    # remove Splus objects from .Data (but not help files)
    # In particular, get rid of __nonfi* file (installFromSFiles()
    # can remove all objects, but not the __nonfi* file).
    # If is fine if .Data does not exist, but if it does
    # it must be writable.
    my $self = shift ;
    my $destdir = $self->{destdir} ;
    my $dot_data = "$destdir/.Data" ;
    my $meta = "$dot_data/__Meta" ;
    foreach my $dir ($meta, $dot_data) {
        if (-d "$dir") {
            -w "$dir" or die "Directory " . abs_path("$dir") . " is not writable" ;
        }
    }
    foreach my $dir ($meta, $dot_data) {
        if (-d "$dir") {
            $self->{verbose} and print "$0: removing files from " . abs_path("$dir") . "\n" ;
            opendir my $dirhandle, $dir or die "Cannot open $dir ($!)";
            while (my $filename = readdir $dirhandle) {
                next if $filename =~ /^(\.|\.\.|\.Audit)$/ ;
                my $fullname = "$dir/$filename" ;
                next if -d $fullname ;
                unlink $fullname or warn "Could not remove file $fullname, .Data may not be clean" ;
            }
            closedir $dirhandle ;
        }
    }
}

sub _canonicalize_destdir
{
    # not for end-user use
    # No arguments (except implicit $self).
    # Look at destdir and remove possible trailing /.Data
    # Also, change any backslashes to slashes.
    # This will only be used by perl and Splus, not by cmd.exe.
    my $self = shift ;
    $self->{destdir} =~ s^\\^/^g ;
    $self->{destdir} =~ s^/\.Data$^^i ;
    -d $self->{destdir} or die "--destdir $self->{destdir} does not name a directory" ;
}

sub add_source_files
{
    my $self = shift ;
    foreach my $arg (@_) {
        foreach my $source_file (glob($arg)) {
           -e $source_file or _Syntax "data file $source_file does not exist" ;
           -d $source_file and _Syntax "Putative data file $source_file is a directory (use --destdir dir)" ;
           ${$self->{srcs}}{"$source_file"}=$self->{src_id}++ ;
        }
    }
}
sub delete_source_files
{
    my $self = shift ;
    foreach my $source_file (@_) {
        delete ${$self->{srcs}}{"$source_file"} ;
    }
}

sub get_source_files
{
    my $self = shift ;
    # sort to order in which they were given
    my @sorted_srcs = sort { ${$self->{srcs}}{$a} <=> ${$self->{srcs}}{$b} } keys(%{$self->{srcs}}) ;
    @sorted_srcs ;
}

sub make
{
    my $self = shift ;
    my @srcs = get_source_files($self) ;
    @srcs = map { $_ =~ s^\\^/^g ; $_ } @srcs ; # To avoid Splus backslash issues on Windows (this is only used in S code)
    if ( scalar @srcs == 0) {
        print "$0 : No data files to install\n" ;
    } else {
        print "Installing data files to directory " . abs_path($self->{destdir}) . " from the ", scalar @srcs, " file(s) ", join(", ", @srcs), "\n" ;
        $self->_prepare_destdir() ;
        $self->_clean_destdir_data() if $self->{clean_first} ;
        my $args = "c('" . join("', '", @srcs) . "')" ;
        $args .= ", where='" . $self->{destdir} . "'" ;
        $args .= ", clean.first=TRUE" if $self->{clean_first} ;
        my $descriptionFile = "../DESCRIPTION" ;
        $args .= ", descriptionFile='$descriptionFile'" if -e $descriptionFile ;
        my $cmd = "installFromDataFiles(" . $args . ");" ;
	$self->{verbose} and print "$0: Splus cmd=$cmd\n";
        my $had_dot_data = -d ".Data" ;
        my @out = Splus_run($cmd, "-vanilla -quitonerror");
        if (!$had_dot_data && -d ".Data") {
            # work around bug on Windows: Splus START -vanilla  makes .Data
            rm_empty_dot_data();
        }
        print "$0: Splus output:\n", join("\n\t", @out), "\n";
        # Splus_run_ex would return exit status, but Splus on Windows
        # always gives exit status of 0.  Hence we grep for string
        # that quitonerror produces.  Also, installFromDataFiles()
        # only gives warnings when it cannot process a file.  This
        # should be changed, but in the meantime we catch the warning.
        map { die("Error installing data") if /^Quitting Splus session because of error$/ || /Could not install/ || /^Terminating S Session:/ } @out ;
    }
    1 ; # should return status indicator, or die if make failed
}

1;
