package Splus::src2bin ;
use v5.8;
use strict ;

=head1 NAME
  Splus::src2bin - convert a source package into a binary
=head1 SYNOPSIS
  use Splus::src2bin;
  $sb = Splus::src2bin->new(qw(--directory pkg_dir
                               --parent parent_directory
                               --name name
                               --verbose
                               --clean-first
                               --update-description
                               --no-sinstall
                               --no-shlib
                               --no-helpinstall
                               --no-datainstall
                               --no-javainstall
                               --no-instinstall))
  # none of the above args are needed if your cwd is pkg_dir
  $sb->make() ; # do everything asked for
  $sb->do_sinstall() ;
  $sb->do_helpinstall() ;
  $sb->do_shlib() ;
  $sb->do_datainstall() ;
  $sb->do_javainstall() ;
  $sb->do_swinguiinstall() ;
  $sb->set_no_sinstall() ;
  $sb->set_no_helpinstall() ;
  $sb->set_no_shlib() ;
  $sb->set_no_datainstall() ;
  $sb->set_no_swinguiinstall() ;
=cut

use Splus::Vars;
use Splus::SplusUtils ;
use Splus::SHLIB ;
use Splus::SINSTALL ;
use Splus::DATAINSTALL ;
use Splus::JAVAINSTALL ;
use Splus::HELPINSTALL ;
use Splus::SWINGUIINSTALL ;
use Cwd qw(getcwd abs_path) ;
use File::Path ;
use File::Basename ;

Splus::Vars::error("SHOME");
my $SHOME=${Splus::Vars::SHOME} ;

sub syntax {
    print "-h,--help             # print this help message\n";
    print "--parent parentDir    # parent directory of package source\n";
    print "--name packageName    # parentDir/packageName contains the package itself\n";
    print "--directory directory # parentDir/packageName as one argument (default=.)\n";
    print "If neither --parent nor --name given, then parent is dirname cwd, name is basename cwd\n";
    print "-v,--verbose          # give more detailed progress reports\n";
    print "--clean-first         # delete old binary objects before making new ones\n";
    print "--clean-after         # delete unneeded files after the build\n";
    print "--update-description  # add Built: line to DESCRIPTION file.\n";
    print "--no-clean-after      # do not delete unneeded files after the build\n";
    print "--no-sinstall         # don't install S functions from R directory\n" ;
    print "--no-shlib            # don't compile code from src directory to make shared/dynamic library\n" ;
    print "--no-helpinstall      # don't install help files from man directory\n" ;
    print "--no-datainstall      # don't install data files from data directory\n" ;
    print "--no-javainstall      # don't install jar files from java directory\n" ;
    print "--no-instinstall      # don't install files from inst directory\n" ;
    print "--no-swinguiinstall   # don't install files from swingui directory\n" ;
    # help calls this with no arguments, problems invoke it with an error string.
    @_ && die join(", ", @_) ;
}

sub new {
    my $class = shift ;
    my $self = {} ;
    my @args = globlist(@_) ; # *.c -> a.c b.c ...
    $self->{verbose} = 0 ;
    $self->{clean_after} = 0 ;
    while (@args) {
        my $arg = shift @args ;
        if ($arg =~ /^-h|--help/) {
            syntax() ;
            exit() ;
        } elsif ($arg =~ /^-p|--parent/) {
            $arg = shift @args or die "No directory name after --parent argument" ;
            $self->{parent_dir} = abs_path($arg) ;
        } elsif ($arg =~ /^-n|--name/) {
            $arg = shift @args or syntax("No package name after --name argument") ;
            $self->{name} = $arg ;
        } elsif ($arg =~ /^-d|--dir/) {
            $arg = shift @args or syntax("No directory name after --directory argument") ;
            $self->{dir} = abs_path($arg) ;
        } elsif ($arg =~ /^-v|--verbose/) {
            $self->{verbose} = 1 ;
        } elsif ($arg =~ /^--clean-first/) {
            $self->{clean_first} = 1 ;
        } elsif ($arg =~ /^--clean-after/) {
            $self->{clean_after} = 1 ;
        } elsif ($arg =~ /^--update-description/) {
            $self->{update_description} = 1 ;
        } elsif ($arg =~ /^--no-clean-after/) {
            $self->{clean_after} = 0 ;
        } elsif ($arg =~ /^--no-sinstall/) {
            $self->{no_sinstall} = 1 ;
        } elsif ($arg =~ /^--no-shlib/) {
            $self->{no_shlib} = 1 ;
        } elsif ($arg =~ /^--no-helpinstall/) {
            $self->{no_helpinstall} = 1 ;
        } elsif ($arg =~ /^--no-datainstall/) {
            $self->{no_datainstall} = 1 ;
        } elsif ($arg =~ /^--no-javainstall/) {
            $self->{no_javainstall} = 1 ;
        } elsif ($arg =~ /^--no-instinstall/) {
            $self->{no_instinstall} = 1 ;
        } elsif ($arg =~ /^--no-swinguiinstall/) {
            $self->{no_swinguiinstall} = 1 ;
        } else {
            syntax("Argument $arg not recognized") ;
        }
    }
    # now take care of defaults
    if ($self->{name}) {
         $self->{parent_dir} or $self->{parent_dir} = abs_path(".") ;
         if ($self->{dir}) {
             syntax("Gave both --directory and --name, did you mean --parent and --name?");
         }
         $self->{dir} = abs_path("$self->{parent_dir}/$self->{name}") ;
    }
    $self->{dir} or $self->{dir} = abs_path(".") ;
    $self->{name} or $self->{name} = basename($self->{dir}) ;
    $self->{parent_dir} or $self->{parent_dir} = dirname($self->{dir}) ;
    # sanity checks
    -d $self->{dir} or die "$self->{dir} is not a directory" ;
    -e "$self->{dir}/DESCRIPTION" or warn "No DESCRIPTION file found in $self->{dir}" ;
    $self->{no_shlib} |= ! -d "$self->{dir}/src" ;
    $self->{no_datainstall} |= ! -d "$self->{dir}/data" ;
    $self->{no_javainstall} |= ! -d "$self->{dir}/java" ;
    $self->{no_instinstall} |= ! -d "$self->{dir}/inst" ;
    $self->{no_swinguiinstall} |= ! -d "$self->{dir}/swingui" ;
    $self->{no_swinguiinstall} |= ${Splus::Vars::OSTYPE} ne "windows" ;
    # 
    if ($self->{verbose}) {
        print "name=$self->{name}\n";
        print "parent=$self->{parent_dir}\n";
        print "dir=$self->{dir}\n" ;
    }
    # bless things and return
    bless $self, $class ;
}

sub make
{
    my $self = shift ;
    $self->do_sinstall() if ! $self->{no_sinstall} ; 
    $self->do_shlib() if ! $self->{no_shlib} ;
    $self->do_datainstall() if ! $self->{no_datainstall} ;
    $self->do_javainstall() if ! $self->{no_javainstall} ;
    $self->do_helpinstall() if ! $self->{no_helpinstall} ;
    $self->do_instinstall() if ! $self->{no_instinstall} ;
    $self->do_swinguiinstall() if ! $self->{no_swinguiinstall} ;
    # Make .Data, etc., directory if no one else did
    # We need to be able to attach the package, even if
    # it has no functions, datasets, or help files:  its
    # inst directory may contain something usefule, like PDF files.
    if (! -d "$self->{dir}/.Data") {
        foreach my $subdir (qw(.Data .Data/__Meta .Data/__Hhelp .Data/__Shelp)) {
            if (! -d "$self->{dir}/$subdir") {
                 mkdir "$self->{dir}/$subdir" or die "Cannot make directory $self->{dir}/$subdir" ;
            }
        }
    }
    add_built_stamp_to_description_file($self->{parent_dir} . "/" . $self->{name} . "/DESCRIPTION") if $self->{update_description} ;
}

sub do_shlib {
    my $self = shift ;
    my $subdir = "src" ;
    if ($self->{verbose}) { print "* Making dynamic/shared library for package $self->{name}\n" ; }
    my $cwd = getcwd() ;
    eval { 
       chdir "$self->{dir}/$subdir" or die "Cannot chdir $self->{dir}/$subdir ($!)" ;
       my @args = qw(-o ../S) ;
       push @args, "--verbose" if $self->{verbose} ;
       push @args, "--clean-first" if $self->{clean_first} ;
       push @args, "--clean-after" if $self->{clean_after} ;
       my $sh = Splus::SHLIB->new(@args) ;
       $sh->make() ;
    } ;
    die "Problem in SHLIB: $@" if $@ ;
    chdir $cwd or die "Cannot return to directory $cwd after SHLIB in $self->{dir}/$subdir" ;
    if ($self->{verbose}) { print "  Done making dynamic/shared library for package $self->{name}\n" ; }
}

sub do_sinstall {
    my $self = shift ;
    my $subdir = "R" ;
    if ($self->{verbose}) { print "* Installing S functions for package $self->{name}\n" }
    if (! -d "$self->{dir}/$subdir") {
       print "   No $subdir directory in this package (OK)\n" if $self->{verbose} ;
       return ;
    }
    my $cwd = getcwd() ;
    eval { 
       chdir "$self->{dir}/$subdir" or die "Cannot chdir $self->{dir}/$subdir ($!)" ;
       my @args = qw(--destdir ..) ;
       push @args, "--verbose" if $self->{verbose} ;
       push @args, "--clean-first" if $self->{clean_first} && !$self->{cleaned_dot_data} ;
       push @args, "--clean-after" if $self->{clean_after} ;
       $self->{cleaned_dot_data} = 1 ;
       my $sh = Splus::SINSTALL->new(@args) ;
       $sh->make() ;
    } ;
    die "Problem in SINSTALL: $@" if $@ ;
    chdir $cwd or die "Cannot return to directory $cwd after SINSTALL in $self->{dir}/$subdir" ;
    if ($self->{verbose}) { print "  Done installing S functions for package $self->{name}\n" ; }
}

sub do_helpinstall {
    my $self = shift ;
    my $subdir = "man" ;
    if ($self->{verbose}) { print "* Installing help files for package $self->{name}\n" ; }
    if (! -d "$self->{dir}/$subdir") {
       warn "   No $subdir directory in package $self->{name}!\n" ;
       return ;
    }
    my $cwd = getcwd() ;
    eval { 
       chdir "$self->{dir}/$subdir" or die "Cannot chdir $self->{dir}/$subdir ($!)" ;
       my @args = qw(--destdir ..) ;
       push @args, "--verbose" if $self->{verbose} ;
       push @args, "--clean-first" if $self->{clean_first} ;
       push @args, "--clean-after" if $self->{clean_after} ;
       print "  Calling HELPINSTALL " . join(" ", @args) . "\n";
       my $hp = Splus::HELPINSTALL->new(@args) ;
       $hp->make() ;
    } ;
    die "Problem in HELPINSTALL: $@" if $@ ;
    chdir $cwd or die "Cannot return to directory $cwd after helpinstall in $self->{dir}/$subdir" ;
    if ($self->{verbose}) { print "  Done installing help files for package $self->{name}\n" ; }
}

sub do_datainstall {
    my $self = shift ;
    my $subdir = "data" ;
    if ($self->{verbose}) { print "* Installing data files for package $self->{name}\n" }
    if (! -d "$self->{dir}/$subdir") {
       print "   No $subdir directory in this package (OK)\n" if $self->{verbose} ;
       return ;
    }
    my $cwd = getcwd() ;
    eval { 
       chdir "$self->{dir}/$subdir" or die "Cannot chdir $self->{dir}/$subdir ($!)" ;
       my @args = qw(--destdir ..) ;
       push @args, "--verbose" if $self->{verbose} ;
       push @args, "--clean-first" if $self->{clean_first} && !$self->{cleaned_dot_data} ;
       push @args, "--clean-after" if $self->{clean_after} ;
       $self->{cleaned_dot_data} = 0 ;
       print "  Calling datainstall " . join(" ", @args) . "\n";
       my $dt = Splus::DATAINSTALL->new(@args) ;
       $dt->make() ;
    } ; # need semicolon after eval's {}
    die "Problem in DATAINSTALL: $@" if $@ ;
    chdir $cwd or die "Cannot return to directory $cwd after datainstall in $self->{dir}/$subdir" ;
    if ($self->{verbose}) { print "  Done installing data files for package $self->{name}\n" ; }
}

sub do_javainstall {
    my $self = shift ;
    my $subdir = "java" ;
    if ($self->{verbose}) { print "* Compiling Java files for package $self->{name}\n" }
    if (! -d "$self->{dir}/$subdir") {
       print "   No $subdir directory in this package (OK)\n" if $self->{verbose} ;
       return ;
    }
    my $cwd = getcwd() ;
    eval { 
       chdir "$self->{dir}/$subdir" or die "Cannot chdir $self->{dir}/$subdir ($!)" ;
       my @args = qw(--destdir .. --name) ;
       push @args, $self->{name} ;
       my $jv = Splus::JAVAINSTALL->new(@args) ;
       $jv->make() ;
    } ; # need semicolon after eval's {}
    die "Problem in JAVAINSTALL: $@" if $@ ;
    chdir $cwd or die "Cannot return to directory $cwd after javainstall in $self->{dir}/$subdir" ;
    if ($self->{verbose}) { print "  Done compiling java files for package $self->{name}\n" ; }
}

sub do_instinstall {
    my $self = shift ;
    my $subdir = "inst" ;
    if ($self->{verbose}) { print "* Installing inst files for package $self->{name}\n" }
    if (! -d "$self->{dir}/$subdir") {
       print "   No $subdir directory in this package (OK)\n" if $self->{verbose} ;
       return ;
    }
    opendir my $dirhandle, "$self->{dir}/$subdir" or die "Cannot open $self->{dir}/$subdir to copy files from it ($!)";
    while (my $basefilename = readdir $dirhandle) {
        next if $basefilename =~ /^(\.|\.\.)$/ ;
        my $srcname = "$self->{dir}/$subdir/$basefilename" ;
        next if (-l $srcname) ; # no symbolic links copied
        my $destname = "$self->{dir}/$basefilename" ;
        rmtree($destname, $self->{verbose}, 1) if -e $destname ;
        print "  Copying $basefilename from $self->{dir}/$subdir to $self->{dir}\n" ;
        copy_recursive($srcname, $self->{dir}) ;
    }
    closedir $dirhandle ;
    if ($self->{verbose}) { print "  Done installing files from inst/ for package $self->{name}\n" ; }
}

sub do_swinguiinstall {
    my $self = shift ;
    my $subdir = "swingui" ;
    my $ostype = ${Splus::Vars::OSTYPE} ; # "windows" or "unix"
    return if $ostype != "windows" ;
    if ($self->{verbose}) { print "* Installing $subdir files for package $self->{name}\n" }
    if (! -d "$self->{dir}/$subdir") {
       print "   No $subdir directory in this package (OK)\n" if $self->{verbose} ;
       return ;
    }
    my $cwd = getcwd() ;
    eval { 
       chdir "$self->{dir}/$subdir" or die "Cannot chdir $self->{dir}/$subdir ($!)" ;
       my @args = qw(--destdir .. --name) ;
       push @args, $self->{name} ;
       my $swinguiinstall = Splus::SWINGUIINSTALL->new(@args) ;
       $swinguiinstall->make() ;
    } ; # need semicolon after eval's {}
    my $eval_status=$@ ;
    chdir $cwd or die "Cannot return to directory $cwd after swinguiinstall in $self->{dir}/$subdir" ;
    die "Problem in SWINGUIINSTALL: $@" if $eval_status ;
    if ($self->{verbose}) { print "  Done installing files from $subdir for package $self->{name}\n" ; }
}

1; 
